/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search.lucene;

import com.arsdigita.search.QueryEngine;
import com.arsdigita.search.Search;
import com.arsdigita.search.IndexerType;
import com.arsdigita.search.BaseDocument;
import com.arsdigita.search.ResultSet;
import com.arsdigita.search.QuerySpecification;
import com.arsdigita.search.FilterSpecification;
import com.arsdigita.search.FilterType;

import com.arsdigita.persistence.OID;

import com.arsdigita.util.Assert;

import java.util.Date;
import java.util.Iterator;
import java.util.Locale;
import java.util.Collections;

import java.math.BigDecimal;

import org.apache.log4j.Logger;


class LuceneResultSet implements ResultSet {

    private static final Logger s_log = 
        Logger.getLogger(LuceneResultSet.class);
    
    private LuceneSearch m_search;
    private long m_count;
    private long m_queryTime;

    public LuceneResultSet(LuceneSearch search) {
        m_search = search;
        Date start = new Date();
        m_count = search.size();
        Date end = new Date();
        
        m_queryTime = end.getTime() - start.getTime();
        
        if (s_log.isDebugEnabled()) {
            s_log.debug("Result set count is " + m_count + 
                        ", query duration " + m_queryTime);
        }
    }

    public Iterator getDocuments(long offset,
                                 long count) {
        if (s_log.isDebugEnabled()) {
            s_log.debug("Paginating at offset " + offset + 
                        " for " + count + " rows");
        }
        if (count == 0) {
            return Collections.EMPTY_LIST.iterator();
        }

        return new LuceneIterator(m_search,
                                  offset,
                                  count);
    }

    public void close() {
        if (s_log.isDebugEnabled()) {
            s_log.debug("Closing search result set");
        }
        m_search.close();
    }
    
    public String getEngine() {
        return IndexerType.LUCENE.getKey();
    }

    public long getCount() {
        return m_count;
    }
    
    public long getQueryTime() {
        return m_queryTime;
    }
    
    private class LuceneIterator implements Iterator {
        
        private LuceneSearch m_search;
        private long m_count;
        private boolean m_hasNext;

        public LuceneIterator(LuceneSearch search,
                              long offset,
                              long count) {
            m_search = search;
            m_count = count;
            do {
                if (s_log.isDebugEnabled()) {
                    s_log.debug("Peek object " + offset);
                }
                m_hasNext = m_search.next();
            } while (offset-- > 0);
        }
        
        public boolean hasNext() {
            return m_hasNext;
        }
        
        public Object next() {
            Assert.truth(m_hasNext, "hasNext");

            if (s_log.isDebugEnabled()) {
                s_log.debug("Got next object");
            }

            BaseDocument result = new BaseDocument(
                new OID(m_search.getType(),
                        m_search.getID()),
                m_search.getLocale(),
                m_search.getTitle(),
                m_search.getSummary(),
                m_search.getCreationDate(),
                null,
                m_search.getLastModifiedDate(),
                null,
                new BigDecimal(m_search.getScore() * 100.0).setScale(0,BigDecimal.ROUND_HALF_UP)
            );

            // Prepare next result.
            if (s_log.isDebugEnabled()) {
                s_log.debug("Peeking at result count is " + m_count);
            }

            if (m_count-- > 0) {
                m_hasNext = m_search.next();
            } else {
                m_hasNext = false;
            }
            return result;
        }
        
        public void remove() {
            throw new UnsupportedOperationException("cannot remove items");
        }
    }    
}
