/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search.lucene;

import com.arsdigita.initializer.Configuration;
import com.arsdigita.initializer.InitializationException;
import com.arsdigita.search.FilterType;
import com.arsdigita.search.IndexerType;
import com.arsdigita.search.QueryEngineRegistry;
import com.arsdigita.search.Search;
import com.arsdigita.search.filters.ObjectTypeFilterType;

import java.io.File;
import java.io.IOException;
import java.util.Date;

import org.apache.log4j.Logger;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;

/**
 * Initializer
 *
 * Initializes the Lucene package
 *
 * @author Richard Su (richard.su@alum.mit.edu)
 * @version $Id: //core-platform/dev/src/com/arsdigita/search/lucene/Initializer.java#7 $
 *
 */
public class Initializer implements com.arsdigita.initializer.Initializer {

    private static final Logger LOG = Logger.getLogger(Initializer.class);

    public final static Loader LOADER = new IndexId.LoaderImpl();

    private Configuration m_conf = new Configuration();

    public Initializer() {}

    public Configuration getConfiguration() {
        return m_conf;
    }

    /*
     * Called on startup.
     * @throws InitializationException
     */
    public void startup() {
        if (Search.getConfig().isLuceneEnabled()) {
            LuceneConfig conf = LuceneConfig.getConfig();
            String location = conf.getIndexLocation();
            int interval = conf.getIndexerInterval();
            LOG.info("Lucene index location: " + location);

            try {
                if (!IndexReader.indexExists(location)) {
                    File f = new File(location);
                    f.mkdirs();
                    IndexWriter iw = new IndexWriter
                        (location, new StandardAnalyzer(), true);
                    iw.close();
                    LOG.info("Lucene created index directory");
                }
            } catch (IOException ex) {
                throw new InitializationException
                    ("lucene index id has not been initialized", ex);
            }

            Index.setLocation(location);
            // FIXME: This is a hack.  Delay the start of the timer task by 5
            // seconds and keep your fingers crosses in hopes that
            // Index.getIndexerID() will be ready to return a valid value by
            // then.  -- vadimn@redhat.com, 2004-01-07
            Date delayedStart = new Date(System.currentTimeMillis() + 5000L);
            Index.getTimer().schedule
                (new Indexer(Index.getLocation()),
                 delayedStart,
                 ((long)interval * 1000l));

            QueryEngineRegistry.registerEngine
                (IndexerType.LUCENE, new FilterType[] {},
                 new BaseQueryEngine());
            QueryEngineRegistry.registerEngine
                (IndexerType.LUCENE, new FilterType[] {
                    new ObjectTypeFilterType(),
                }, new BaseQueryEngine());
        }
    }

    /**
     * Called on shutdown. It's probably not a good idea to depend on this
     * being called.
     **/
    public void shutdown() {
    }

    /**
     * Only used by <code>CoreLoader</code>.
     **/
    public interface Loader {
        void load();
    }
}
