/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search.filters;

import com.arsdigita.search.FilterSpecification;
import com.arsdigita.kernel.Party;
import com.arsdigita.kernel.permissions.PermissionManager;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.persistence.OID;


/**
 * A filter specification for supplying a party
 * and privilege to filter against.
 */
public class PermissionFilterSpecification extends FilterSpecification {
    
    public final static String PARTY = "party";
    public final static String PRIVILEGE = "privilege";

    /**
     * Creates a new permission filter spec with
     * 'The Public' party and the READ privilege
     */
    public PermissionFilterSpecification() {
        this(lookupPublic());
    }
    
    /**
     * Creates a new permission filter spec for a
     * party with READ privilege
     * @param party the party to check permissions for
     */
    public PermissionFilterSpecification(Party party) {
        this(party, PrivilegeDescriptor.READ);
    }
    
    /**
     * Creates a new permission filter spec for a
     * party with a specified privilege
     * @param party the party to check permissions for
     * @param privilege the privilege to check
     */
    public PermissionFilterSpecification(Party party,
                                         PrivilegeDescriptor privilege) {
        super(new Object[] { PARTY, party, PRIVILEGE, privilege },
              new PermissionFilterType());
    }
    
    /**
     * Returns the party to check permissions for
     * @return the party to check
     */
    public Party getParty() {
        return (Party)get(PARTY);
    }
    
    /**
     * Returns the privilege to check against
     * @return the privilege
     */
    public PrivilegeDescriptor getPrivilege() {
        return (PrivilegeDescriptor)get(PRIVILEGE);
    }

    private static Party lookupPublic() {
        OID oid = new OID(Party.BASE_DATA_OBJECT_TYPE,
                          PermissionManager.VIRTUAL_PUBLIC_ID);
        
        return (Party)DomainObjectFactory.newInstance(oid);
    }
}
