/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search;

import com.arsdigita.persistence.OID;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.kernel.Party;
import java.util.Date;
import java.util.Locale;


/**
 * The interface defining an adapter between a domain object
 * and the search index metadata. Any domain object which
 * wishes to make itself searchable should register an
 * instance of this interface with the MetadataProviderRegistry
 * 
 * @see com.arsdigita.search.MetadataProviderRegistry
 * @see com.arsdigita.search.ContentProvider
 */
public interface MetadataProvider {
    
    /**
     * Gets an opaque fragment of object type specific metadata.
     * This method is a short term workaround for limitations
     * in our Lucene implementation. ie for c.a.cms.ContentPage
     * it lets us distinguish between live and draft items. It
     * will likely be removed in the near future. 
     *
     * @param dobj the domain object
     * @return the type specific metadata
     *
     * @pos $retval != null
     */
    String getTypeSpecificInfo(DomainObject dobj);
    
    /**
     * Gets the locale to which this object belongs
     *
     * @param dobj the domain object
     * @return the locale of the object
     *
     * @pos $retval != null
     */
    Locale getLocale(DomainObject dobj);

    /**
     * Gets the Title property for the DomainObject
     *
     * @param dobj the domain object
     * @return title of the object
     *
     * @post $retval != null
     */
    String getTitle(DomainObject dobj);
    
    /**
     * Gets the (optional) summary of the DomainObject
     *
     * @param dobj the domain object
     * @return the object summary, or null
     */
    String getSummary(DomainObject dobj);
    
    /**
     * Gets the (optional) creation date of the DomainObject
     *
     * @param dobj the domain object
     * @return the creation date, or null
     */
    Date getCreationDate(DomainObject dobj);

    /**
     * Gets the (optional) creating party of the DomainObject
     *
     * @param dobj the domain object
     * @return the creation party, or null
     */
    Party getCreationParty(DomainObject dobj);

    /**
     * Gets the (optional) last modification date of the DomainObject
     *
     * @param dobj the domain object
     * @return the modification date, or null
     */
    Date getLastModifiedDate(DomainObject dobj);

    /**
     * Gets the (optional) last modifying party of the DomainObject
     *
     * @param dobj the domain object
     * @return the modification party, or null
     */
    Party getLastModifiedParty(DomainObject dobj);

    /**
     * Gets the content for the DomainObject.
     * Content can be returned in multiple formats.
     * To see which the current indexer supports
     * use Search.getConfig().allowsXXXContent()
     *
     * @param dobj the domain object
     * @return array of content for this object, or an empty array
     * @post $retval != null
     */
    ContentProvider[] getContent(DomainObject dobj,
                                 ContentType type);
    
}
