/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.runtime;

import com.arsdigita.persistence.pdl.PDLCompiler;
import com.arsdigita.persistence.pdl.PDLSource;


/**
 * The PDLInitializer class provides an implementation of the {@link
 * Initializer} interface that works in conjunction with the {@link
 * PDLSource} interface and implementations thereof in order to
 * provide a convenient way to initialize object-relational metadata.
 * This class is most convenient when used in conjunction with the
 * {@link CompoundInitializer} class. For example:
 *
 * <blockquote>
 * package com.arsdigita.exampleApp;
 *
 * public class Initializer extends CompoundInitializer {
 *
 *     public Initializer() {
 *         final String url = RuntimeConfig.getConfig().getJDBCURL();
 *         final int database = DbHelper.getDatabaseFromURL(url);
 *
 *         add(new PDLInitializer
 *             (new ManifestSource
 *              ("example-app.pdl.mf",
 *               new NameFilter(DbHelper.getDatabaseSuffix(database),
 *                              "pdl"))));
 *     }
 *
 * }
 * </blockquote>
 *
 * The class defined above will load all the object-relational
 * metadata for the current database referenced from the
 * example-app.pdl.mf manifest file.
 *
 * @author Rafael H. Schloming &lt;rhs@mit.edu&gt;
 * @version $Revision: #5 $ $Date: 2004/04/07 $
 **/

public class PDLInitializer implements Initializer {

    public final static String versionId = "$Id: //core-platform/dev/src/com/arsdigita/runtime/PDLInitializer.java#5 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private final PDLSource m_source;

    /**
     * Constructs a new implementation of the {@link Initializer}
     * interface that will upon invokation load any object-relational
     * metadata referenced by the given PDLSource.
     *
     * @param source The object-relational metadata to load.
     **/

    public PDLInitializer(PDLSource source) {
        m_source = source;
    }

    /**
     * An implementation of the {@link
     * Initializer#init(DataInitEvent)} that loads the
     * object-relational metadata from the PDLSource specified during
     * construction of this PDLInitializer.
     **/

    public void init(DataInitEvent evt) {
        PDLCompiler compiler = evt.getCompiler();
        m_source.parse(compiler);
    }

    /**
     * An empty implementation of {@link Initializer#init(DomainInitEvent)}.
     **/

    public void init(DomainInitEvent evt) {
        // do nothing
    }

    /**
     * An empty implementation of {@link Initializer#init(LegacyInitEvent)}.
     **/

    public void init(LegacyInitEvent evt) {
        // do nothing
    }

}
