/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.packaging;

import com.arsdigita.persistence.*;
import com.arsdigita.runtime.*;
import com.arsdigita.util.*;

import com.arsdigita.xml.XML;
import org.xml.sax.Attributes;
import org.xml.sax.helpers.DefaultHandler;

import java.io.*;
import java.util.*;

/**
 * Checklist
 *
 * @author Rafael H. Schloming &lt;rhs@mit.edu&gt;
 * @version $Revision: #7 $ $Date: 2004/04/07 $
 **/

class Checklist {

    public final static String versionId = "$Id: //core-platform/dev/src/com/arsdigita/packaging/Checklist.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    public static Checklist get(String pkg) {
        ClassLoader ldr = Checklist.class.getClassLoader();
        InputStream is = ldr.getResourceAsStream(pkg + ".checklist");
        if (is == null) { return null; }
        Checklist list = new Checklist(is);
        try { is.close(); }
        catch (IOException e) { throw new UncheckedWrapperException(e); }
        return list;
    }

    public static final int SCHEMA = 0;
    public static final int DATA = 1;
    public static final int STARTUP = 2;

    private List m_groups = new ArrayList();

    public Checklist(InputStream is) {
        XML.parse(is, new ChecklistHandler());
    }

    public boolean run(int type, ScriptContext ctx) {
        boolean result = true;
        for (Iterator it = m_groups.iterator(); it.hasNext(); ) {
            Group group = (Group) it.next();
            if (group.getType() == type) {
                result &= group.run(ctx);
            }
        }
        return result;
    }

    private class Group {

        private int m_type;
        private List m_checks;

        public Group(int type) {
            m_type = type;
            m_checks = new ArrayList();
        }

        public int getType() {
            return m_type;
        }

        void addCheck(Check check) {
            m_checks.add(check);
        }

        public List getChecks() {
            return m_checks;
        }

        public boolean run(ScriptContext ctx) {
            for (Iterator it = m_checks.iterator(); it.hasNext(); ) {
                Check check = (Check) it.next();
                check.run(ctx);
                Check.Status status = check.getStatus();
                if (status == null) {
                    throw new IllegalStateException
                        (check.getClass().getName() + ": check failed to " +
                         "report exit status");
                }
                if (status.equals(Check.FAIL)) {
                    return false;
                }
            }

            return true;
        }

    }

    private static final String CHECKS = "checks";
    private static final String TYPE = "type";
    private static final String CHECK = "check";
    private static final String CLASS = "class";

    private class ChecklistHandler extends DefaultHandler {

        private Group m_group = null;

        public void startElement(String uri, String name, String qn,
                                 Attributes attrs) {
            if (name.equals(CHECKS)) {
                String type = attrs.getValue(uri, TYPE);
                if (type == null) {
                    throw new IllegalStateException
                        (CHECKS + " requires a " + TYPE + " attribute");
                } else if (type.equals("schema")) {
                    m_group = new Group(SCHEMA);
                } else if (type.equals("data")) {
                    m_group = new Group(DATA);
                } else if (type.equals("startup")) {
                    m_group = new Group(STARTUP);
                } else {
                    throw new IllegalStateException
                        ("unrecognized value for " + TYPE + " attribute: " +
                         type);
                }
            }

            if (name.equals(CHECK)) {
                if (m_group == null) {
                    throw new IllegalStateException
                        (CHECK + " element cannot appear " +
                         "outside of a " + CHECKS + " group");
                }

                String klass = attrs.getValue(uri, CLASS);
                if (klass == null) {
                    throw new IllegalStateException
                        (CHECK + " element requires a " + CLASS +
                         " attribute");
                }

                m_group.addCheck((Check) Classes.newInstance(klass));
            }
        }

        public void endElement(String uri, String name, String qn) {
            if (name.equals(CHECKS)) {
                m_groups.add(m_group);
                m_group = null;
            }
        }
    }

}
