/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel;

import com.arsdigita.util.StringUtils;
import com.arsdigita.util.Assert;
/**
 * UserFactory is a utility class for creating
 */
public final class UserFactory {

    /**
     * Creates a new user, sets up authentication, and adds to the system. Any parameter
     * not marked Optional must be non-null.
     *
     * @param primaryEmail - The user's primary email
     * @param givenName - First name
     * @param familyName - Last name
     * @param password - The user's password
     * @param passwordQuestion - The question asked of the user if he forgets his password
     * @param passwordAnswer - The answer to the question. Password reset on correct answer
     * @param screenName - AOL, IRC, etc screen name. Optional
     * @param uri - The URI for the user's web page. Optional
     * @param additionalEmail - An additional email address for the user. Optional
     *
     * @return User The user created.
     */
    public static User newUser(EmailAddress primaryEmail,
                               String givenName,
                               String familyName,
                               String password,
                               String passwordQuestion,
                               String passwordAnswer,
                               String screenName,
                               String uri,
                               EmailAddress additionalEmail) {

        Assert.assertNotNull(primaryEmail, "primaryEmail");
        Assert.assertNotNull(givenName, "givenName");
        Assert.assertNotNull(familyName, "familyName");
        Assert.assertNotNull(password, "password");
        Assert.assertNotNull(passwordQuestion, "passwordQuestion");
        Assert.assertNotNull(passwordAnswer, "passwordAnswer");

        User user = new User();
        user.setPrimaryEmail(primaryEmail);

        PersonName name = user.getPersonName();
        name.setGivenName(givenName);
        name.setFamilyName(familyName);

        user.setScreenName(screenName);

        // Check to see if the value has changed from the
        // default.  If not just leave this set to null.

        if (StringUtils.emptyString(uri) == false) {
            user.setURI(uri);
        }

        // Add optional additional email address
        if (null != additionalEmail) {
            user.addEmailAddress(additionalEmail);
        }

        // Make new user persistent
        user.save();
        // Save user authentication credentials.
        UserAuthentication auth =
            UserAuthentication.createForUser(user);

        auth.setPassword(password);
        auth.setPasswordQuestion(passwordQuestion);
        auth.setPasswordAnswer(passwordAnswer);
        auth.save();

        return user;
    }

}
