/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel;

import com.arsdigita.kernel.security.SecurityConfig;
import java.math.BigDecimal;
import java.util.Locale;
import org.apache.log4j.Logger;

/**
 * <p>A central location for commonly used kernel services and their
 * accessories.</p>
 *
 * <p><b>Context.</b> {@link #getContext()} fetches the context record ({@link
 * com.arsdigita.kernel.KernelContext}) of the current thread.</p>
 *
 * <p><b>The system party.</b> {@link #getSystemParty()} returns the
 * party used to perform work as "the system".</p>
 *
 * @author Rafael Schloming
 * @author Richard Li
 * @author Justin Ross
 * @see com.arsdigita.kernel.KernelConfig
 * @see com.arsdigita.kernel.KernelContext
 * @see com.arsdigita.kernel.KernelExcursion
 */
public class Kernel {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/kernel/Kernel.java#17 $" +
        "$Author: mikeb $" +
        "$DateTime: 2004/04/21 16:54:42 $";

    private static final Logger s_log = Logger.getLogger(Kernel.class);

    private static final KernelContext s_initialContext = new KernelContext();

    // The ID of the user that represents "the public", i.e. a non-logged-in user.
    // Created by insert-users.sql.
    private static final BigDecimal PUBLIC_USER_ID = new BigDecimal(-200);
    private static User s_publicUser;

    static {
        s_initialContext.setLocale(Locale.getDefault());
    }

    private static final ThreadLocal s_context = new ThreadLocal() {
            public Object initialValue() {
                return s_initialContext;
            }
        };

    private static final KernelConfig s_config = new KernelConfig();
    private static final SecurityConfig s_securityConfig = new SecurityConfig();

    static {
        s_config.load("ccm-core/kernel.properties");
        s_securityConfig.load("ccm-core/security.properties");
    }

    public static final KernelConfig getConfig() {
        return s_config;
    }

    public static final SecurityConfig getSecurityConfig() {
        return s_securityConfig;
    }

    /**
     * The email address of the built-in system party.
     */
    public static final String SYSTEM_PARTY_EMAIL =
        "acs-system-party@acs-system";
    private static Party s_systemParty = null;

    /**
     * Get the context record of the current thread.
     *
     * @post return != null
     */
    public static final KernelContext getContext() {
        return (KernelContext) s_context.get();
    }

    static final void setContext(KernelContext context) {
        if (s_log.isDebugEnabled()) {
            s_log.debug("Set context to " + context.getDebugInfo());
        }
        s_context.set(context);
    }

    /**
     * Get the system party, the agent of any work the system
     * performs, as apart from what some user or group does. Returns
     * null if the system party is not defined.
     */
    public static final Party getSystemParty() {
        return s_systemParty;
    }

    static final void setSystemParty(Party party) {
        s_systemParty = party;
    }

    /**
     * Get the User that represents "the public", i.e. non-logged-in
     * users.
     **/
    public static final User getPublicUser() {
        if (s_publicUser == null) {
            // We could synchronize this method, but we don't really care if the User
            // object gets loaded more than once.
            s_publicUser = User.retrieve(PUBLIC_USER_ID);
            // Disconnect the object so we can use it across multiple transactions, and
            // so it cannot be modified/deleted.
            s_publicUser.disconnect();
        }

        return s_publicUser;
    }
}
