/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.parameters;

import com.arsdigita.globalization.Globalization;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import javax.servlet.http.HttpServletRequest;



/**
 * A class that represents the model for a time form parameter.
 *
 * @see com.arsdigita.bebop.parameters.DateTimeParameter
 * @author Dave Turner
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/parameters/TimeParameter.java#8 $
 */
public class TimeParameter extends ParameterModel
{

    public TimeParameter ( String name ) {
        super(name);
    }

    /**
     * This method returns a new Calendar object that is manipulated
     * within transformValue to create a Date Object. This method should
     * be overridden if you wish to use a Calendar other than the
     * lenient GregorianCalendar.
     *
     * @param request the servlet request from which Locale can be
     * extracted if needed
     *
     * @return a new Calendar object
     * */
    protected Calendar getCalendar(HttpServletRequest request) {
        return new GregorianCalendar();
    }


    /**
     * Computes a dateTime object from multiple parameters in the
     * request. This method searches for parameters named
     * <code>getName() + ".hour"<code>,
     * <code>getName() + ".minute"<code>,
     * <code>getName() + ".second"<code>, and
     * <code>getName() + ".amOrPm"<code>.
     * */
    public Object transformValue(HttpServletRequest request)
        throws IllegalArgumentException {

        Calendar c = getCalendar(request);
        c.clear();

        String hour = Globalization.decodeParameter(request, getName()+".hour");
        String minute = Globalization.decodeParameter(request, getName()+".minute");
        String second = Globalization.decodeParameter(request, getName()+".second");
        String amOrPm = Globalization.decodeParameter(request, getName()+".amOrPm");

        if ( hour != null ) {
	    int hourInt = Integer.parseInt(hour);
	    if (hourInt == 12) {
		hourInt = 0;
	    }
            c.set(Calendar.HOUR, hourInt);
        }

        if ( minute != null ) {
            c.set(Calendar.MINUTE, Integer.parseInt(minute));
        }

        if ( second != null ) {
            c.set(Calendar.SECOND, Integer.parseInt(second));
        }

        if ( amOrPm != null ) {
            c.set(Calendar.AM_PM, Integer.parseInt(amOrPm));
        }

        return c.getTime();
    }


    public Object unmarshal ( String encoded ) {
        try {
            return new Date(Long.parseLong(encoded));
        } catch ( NumberFormatException ex ) {
            throw new IllegalArgumentException("Cannot unmarshal time '"
                                               + encoded + "': " + ex.getMessage());
        }
    }

    public String marshal ( Object value ) {
        return Long.toString(((Date)value).getTime());
    }

    public Class getValueClass () {
        return Date.class;
    }

}
