/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.list;


/**
 * The abstraction around a list of values used by the {@link
 * com.arsdigita.bebop.List} component. Data sources that need to be
 * displayed with a list component need to implement this
 * interface. The <code>ListModel</code> provides a read-only view of
 * the underlying data. The items in the list model can be accessed by
 * calling {@link #next} repeatedly and accessing the item with calls
 * to {@link #getKey} and {@link #getElement}. Initially, the list
 * model is positioned <i>before</i> the first element and calls to
 * {@link #getKey} and {@link #getElement} should produce a runtime
 * exception until {@link #next} is called for the first time.
 *
 * <p> A <code>ListModel</code> is usually generated for each request that
 * a list component serves. The <code>List</code> component will use its
 * <code>ListModelBuilder</code> to generate a new <code>ListModel</code>
 * whenever it serves a request.
 *
 * <p> The items a list model iterates over are identified by two separate
 * objects: a <i>key</i> that has to be a unique string representation of the
 * actual list item, and an <i>element</i>, which represents the item itself and
 * is passed directly to the {@link ListCellRenderer}. For example, a list model
 * that represents a list of objects that are stored in the database, the key
 * will usually be a suitable representation of the primary key of the object in
 * the database, while the element can be as simple as a string with the "pretty
 * name" of the object. While the element is usually only passed to a
 * <code>ListCellRenderer</code>, the key will often be communicated to other
 * components that need to know which item in the list is currently selected and
 * can be retrieved with a call to {@link
 * com.arsdigita.bebop.List#getSelectedKey(com.arsdigita.bebop.PageState)}. It
 * is therefore important that all the components that need to work with the key
 * generated by a list model agree on how it represents the underlying
 * (database) object.
 *
 * <p> <b>Warning:</b> The signature of <code>getKey</code> will be
 * changed to <code>Obejct getKey()</code> after ACS 4.6 </p>
 *
 * @see com.arsdigita.bebop.List
 * @see ListCellRenderer
 *
 * @author David Lutterkort
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/list/ListModel.java#11 $ */
public interface ListModel {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/list/ListModel.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * Move to the next list item. If there is no next list
     * item, return <code>false</code>. The item's key and element can be
     * accessed with calls to {@link #getKey} and {@link
     * #getElement}. Initially, the list model is positioned <i>before</i>
     * the first item, so that <code>next()</code> has to be called once
     * before it is possible to access the item.
     *
     * @return <code>true</code> if the model is positioned on a valid item
     * after the call returns.  */
    boolean next();

    /**
     * Get the element for the current list item. The
     * concrete type of the object returned is specific to each
     * implementation of <code>ListModel</code> and should be documented
     * there.
     *
     * <p> This method can only be called successfully if the previous cal to
     * {@link #next} returned <code>true</code>.
     *
     * @return the element for the current list item
     * @see #next */
    Object getElement();

    /**
     *  Get the key that
     * uniquely identifies the current list item. The key should be a
     * string that uniquely identifies the list item, at least amongst all
     * items in the list model. If the key needs to be communicated to
     * other components, for example to display details about the list
     * item, all components need to agree how the key identifies the
     * underlying object. For objects stored in a database, this will
     * usually be a suitable string representation for the primary key of
     * the object.
     *
     * <p> This method can only be called successfully if the previous cal to
     * {@link #next} returned <code>true</code>.
     *
     * @return identifies the object underlying the list item uniquely
     * @see #next */
    String getKey();

}
