/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.form;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import com.arsdigita.xml.Element;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.bebop.parameters.StringParameter;


// This interface contains the XML element name of this class
// in a constant which is used when generating XML

/**
 * Displays and manages a WYSIWYG HTML editor that takes advantage of
 * DHTML scripting features. This class uses the open source HTMLArea
 * editor: <a
 * href="http://women.cs.uiuc.edu/htmlarea/">http://women.cs.uiuc.edu/htmlarea/</a>.
 *
 * @author Jim Parsons
 * @author Richard Li
 *
 *    @version $Id: //core-platform/dev/src/com/arsdigita/bebop/form/DHTMLEditor.java#11 $ */
public class DHTMLEditor extends TextArea {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/form/DHTMLEditor.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    /**
     * Constant for specifying <tt>OFF</tt> value for the
     * <tt>WRAP</tt> attribute of this image input.  See <a
     * href="http://developer.netscape.com/docs/manuals/htmlguid/tags10.htm#1340340">here</a>
     * for a description of what this attribute does.  */

    public static final int OFF = 0;

    /**
     * Constant for specifying <tt>HARD</tt> value for the
     * <tt>WRAP</tt> attribute of this image input.  * See <a
     * //href="http://developer.netscape.com/docs/manuals/htmlguid/tags10.htm#1340340">here</a>
     * for a description of what this attribute does.
     */
    public static final int HARD = 1;

    /**
     * Constant for specifying <tt>SOFT</tt> value for the
     * <tt>WRAP</tt> attribute of this image input.  See <a
     * href="http://developer.netscape.com/docs/manuals/htmlguid/tags10.htm#1340340">here</a>
     * for a description of what this attribute does.
     */
    public static final int SOFT = 2;
    
    public static class Config {
        public static final Config STANDARD = new Config("HTMLArea.Config",
                                                         null);

        private String m_name;
        private String m_path;

        public Config(String name) {
            this(name, null);
        }
        public Config(String name,
                      String path) {
            m_name = name;
            m_path = path;
        }
        
        public String getName() {
            return m_name;
        }
        
        public String getPath() {
            return m_path;
        }
        
        public static Config valueOf(String cfg) {
            int offset = cfg.indexOf(",");
            if (offset != -1) {
                return new Config(cfg.substring(0, offset),
                                  cfg.substring(offset+1));
            } else {
                return new Config(cfg);
            }
        }

        public String toString() {
            if (m_path == null) {
                return m_name;
            } else {
                return m_name + "," + m_path;
            }
        }
    }

    private Config m_config;
    private Set m_plugins;

    public DHTMLEditor(String name) {
        this(new StringParameter(name));
    }

    public DHTMLEditor(ParameterModel model) {
        this(model, Config.STANDARD);
    }

    public DHTMLEditor(ParameterModel model,
                       Config config) {
        super(model);
        m_config = config;
        m_plugins = new HashSet();
    }

    /**
     *      Returns a string naming the type of this widget.
     */
    public String getType() {
        return "DHTMLEditor";
    }

    public void setConfig(String config) {
        setAttribute("config", config);
    }

    public void addPlugin(String name) {
        m_plugins.add(name);
    }

    /**
     *      Sets the <tt>ROWS</tt> attribute for the <tt>TEXTAREA</tt> tag.
     */
    public void setRows(int rows) {
        setAttribute("rows", String.valueOf(rows));
    }

    /**
     *      Sets the <tt>COLS</tt> attribute for the <tt>TEXTAREA</tt> tag.
     */
    public void setCols(int cols) {
        setAttribute("cols", String.valueOf(cols));
    }

    /**
     *      Sets the <tt>COLS</tt> attribute for the <tt>TEXTAREA</tt> tag.
     */
    public void setWrap(int wrap) {
        String wrapString = null;

        switch (wrap) {
        case OFF:
            wrapString = "off";
            break;
        case HARD:
            wrapString = "hard";
            break;
        case SOFT:
            wrapString = "soft";
            break;
        }

        if (wrapString != null) {
            setAttribute("wrap", wrapString);
        }
    }

    /**
     * The XML tag.
     *
     * @return The tag to be used for the top level DOM element
     * generated for this type of Widget.
     */
    protected String getElementTag() {
        return BEBOP_DHTMLEDITOR;
    }

    /**
     * Generates the DOM for the DHTML editor widget
     * <p>Generates DOM fragment:
     * <p><code>&lt;bebop:dhtmleditor name=... value=... [onXXX=...]/>
     *  </code>
     */
    public void generateWidget( PageState state, Element parent ) {
        String value = getParameterData(state).marshal();
        Element editor = parent.newChildElement(getElementTag(), BEBOP_XML_NS);

        editor.addAttribute("name", getName());

        if ( value != null ) {
            editor.setText(value);
        } else {
            editor.setText("Edit text here");
        }

        exportAttributes(editor);
        
        Element config = editor.newChildElement("bebop:config", BEBOP_XML_NS);
        config.addAttribute("name", m_config.getName());
        if (m_config.getPath() != null) {
            config.addAttribute("path", m_config.getPath());
        }
        
        Iterator plugins = m_plugins.iterator();
        while (plugins.hasNext()) {
            String name = (String)plugins.next();
            Element plugin = editor.newChildElement("bebop:plugin", BEBOP_XML_NS);
            plugin.addAttribute("name", name);
        }
    }
}
