/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.demo.workflow;

import com.arsdigita.bebop.BoxPanel;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.bebop.Table;
import com.arsdigita.bebop.ToggleLink;
import com.arsdigita.bebop.event.ActionEvent;
import com.arsdigita.bebop.event.ActionListener;
import com.arsdigita.bebop.event.ChangeEvent;
import com.arsdigita.bebop.event.ChangeListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.FormSubmissionListener;
import com.arsdigita.bebop.event.PrintEvent;
import com.arsdigita.bebop.event.PrintListener;
import com.arsdigita.bebop.event.TableActionEvent;
import com.arsdigita.bebop.event.TableActionListener;
import com.arsdigita.bebop.table.DefaultTableCellRenderer;
import com.arsdigita.bebop.table.TableModel;
import com.arsdigita.bebop.table.TableModelBuilder;
import com.arsdigita.util.Assert;
import java.util.Iterator;

/**
 * Describe class <code>WorkflowDisplay</code> here.
 *
 * @author Uday Mathur
 * @author David Lutterkort
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/demo/workflow/ProcessDisplay.java#8 $
 */
public class ProcessDisplay extends BoxPanel
    implements ActionListener, ChangeListener, TableActionListener {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/demo/workflow/ProcessDisplay.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private final SingleSelectionModel m_processes;

    // The "Details for ..." label
    private Label m_title;
    // The list of tasks in the process
    private Table m_taskTable;
    // form to add one more task to the process
    private AddTask m_addTaskForm ;
    // form to edit a task
    private DisplayTask m_displayTask;

    private ToggleLink m_addTask;

    public ProcessDisplay (SingleSelectionModel m) {
        super(VERTICAL, false);

        m_processes = m;
        m_processes.addChangeListener(this);

        m_title = makeTitle();
        add(m_title);

        m_taskTable = makeTaskTable();
        add(m_taskTable);
        m_taskTable.addTableActionListener(this);
        m_taskTable.getRowSelectionModel().addChangeListener(this);

        m_addTask = new ToggleLink("add task");
        add(m_addTask);
        m_addTask.addActionListener(this);

        m_addTaskForm = new AddTask(m_processes);

        m_addTaskForm.addSubmissionListener(new FormSubmissionListener() {
                public void submitted(FormSectionEvent e) {
                    PageState s = e.getPageState();
                    if ( m_addTaskForm.isCancelled(s) ) {
                        m_addTask.setSelected(s, false);
                        m_addTaskForm.setVisible(s, false);
                    }
                }
            });


        add(m_addTaskForm);

        m_displayTask = new DisplayTask(m_taskTable.getRowSelectionModel());
        add(m_displayTask);
    }

    public void actionPerformed(ActionEvent e) {
        PageState s = e.getPageState();
        addTaskMode(s);
    }

    public void cellSelected(TableActionEvent e) {
        PageState s = e.getPageState();

        if ( e.getSource() == m_taskTable ) {
            viewTaskMode(s);
        }
    }

    public void headSelected(TableActionEvent e) {
        return;
    }

    public void stateChanged(ChangeEvent e) {
        PageState s = e.getPageState();

        if ( e.getSource() == m_processes ) {
            m_addTask.setSelected(s, false);
            m_addTaskForm.setVisible(s, false);
            m_taskTable.getRowSelectionModel().clearSelection(s);
            m_displayTask.setVisible(s, false);
        } else if ( e.getSource() == m_taskTable.getRowSelectionModel() ) {
            viewTaskMode(s);
        }
    }

    public void addTaskMode(PageState s) {
        m_taskTable.getRowSelectionModel().clearSelection(s);
        m_displayTask.setVisible(s, false);
        if ( m_addTask.isSelected(s) ) {
            m_addTaskForm.setVisible(s, true);
        }
    }

    public void viewTaskMode(PageState s) {
        if ( m_taskTable.getRowSelectionModel().isSelected(s) ) {
            m_addTask.setSelected(s, false);
            m_addTaskForm.setVisible(s, false);
            m_displayTask.setVisible(s, true);
        } else {
            m_displayTask.setVisible(s, false);
        }
    }

    private Table makeTaskTable() {
        final String[] headers = { "Task", "Depends", "Assignee" };

        TableModelBuilder b = new TableModelBuilder () {
                private boolean m_locked;

                public TableModel makeModel(final Table t, final PageState s) {
                    Assert.assertTrue(m_processes.isSelected(s));

                    return new TableModel() {
                            private Process p =
                                SampleProcesses.getProcess(m_processes.getSelectedKey(s));

                            private int i = -1;

                            public int getColumnCount() {
                                return 3;
                            }

                            public boolean nextRow() {
                                return (++i < p.taskCount());
                            }

                            public Object getElementAt(int columnIndex) {
                                if ( columnIndex == 0 ) {
                                    return p.getTask(i).getName();
                                } else if ( columnIndex == 1 ) {
                                    Task temp_t = (Task) p.getTask(i);
                                    boolean first = true;
                                    StringBuffer temp_b = new StringBuffer(40);
                                    for (Iterator i = temp_t.dependencies(); i.hasNext();
                                         first=false) {
                                        Task dep = (Task) i.next();
                                        if (! first) {
                                            temp_b.append(", ");
                                        }
                                        temp_b.append(dep.getName());
                                    }
                                    return temp_b.toString();
                                } else if ( columnIndex == 2 ) {
                                    return p.getTask(i).getAssignee();
                                }
                                throw new IllegalArgumentException("columnIndex exceeds number of columns available");
                            }

                            public Object getKeyAt(int columnIndex) {
                                return p.getTask(i).getKey();
                            }
                        };
                }

                public void lock() {
                    m_locked = true;
                }

                public final boolean isLocked() {
                    return m_locked;
                }
            };

        Table result = new Table(b, headers);
        result.getColumn(0).setCellRenderer(new DefaultTableCellRenderer(true));
        return result;

    }

    private Label makeTitle() {
        PrintListener l = new PrintListener() {
                public void prepare(PrintEvent e) {
                    Label t = (Label) e.getTarget();
                    PageState s = e.getPageState();
                    Assert.assertTrue(m_processes.isSelected(s));
                    Process p =
                        SampleProcesses.getProcess(m_processes.getSelectedKey(s));
                    t.setLabel("<h4>Details for "
                               + p.getName() + " </h4>");
                    t.setOutputEscaping(false);
                }
            };
        return new Label(l);
    }

}
