/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.demo;

import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.List;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SimpleComponent;
import com.arsdigita.bebop.list.ListCellRenderer;
import com.arsdigita.bebop.list.ListModel;
import com.arsdigita.bebop.list.ListModelBuilder;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.util.LockableImpl;
import com.arsdigita.xml.Element;
import java.math.BigDecimal;
import org.apache.log4j.Logger;

/**
 * A reusable Bebop component to display a list of users that
 * are registered on your site.
 */
public class UserList extends com.arsdigita.bebop.List {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/demo/UserList.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log =
        Logger.getLogger(UserList.class.getName());

    /**
     * A UserList is a List with a model builder that generates a list of
     * the Users registered in the system.
     */
    public UserList() {
        setClassAttr("user-list");
        setModelBuilder(new UserListModelBuilder());
        setCellRenderer(new UserListCellRenderer());
    }

    private class UserListModelBuilder extends LockableImpl
        implements ListModelBuilder {

        /**
         * Returns a ListModel filled from an Oracle query on
         * our users data model tables.
         */
        public ListModel makeModel(List l, PageState ps) {
            UserListModel result = null;
            try {
                DataQuery dq = SessionManager.getSession()
                    .retrieveQuery("com.arsdigita.kernel.RetrieveUsers");
                dq.setParameter("excludeGroupId", new Integer(0));
                result = new UserListModel(dq);
            } catch (Exception e) {
                s_log.error(e);
            }
            return result;
        }
    }

    // The Object that the ListModel.getElement() returns is passed into
    // the ListCellRenderer as the 'value' argument.
    // Hence, the ListCellRenderer first converts its value argument back
    // into a string array (because that is really what
    // UserListModel.getElement) returns.
    // In a more complex application, getElement() would probably return
    // some form of domain object ...

    /**
     * List model representing a list of users.
     */
    private class UserListModel implements ListModel {
        private DataQuery m_dq;

        public UserListModel(DataQuery dq) {
            m_dq = dq;
        }

        public boolean next() {
            boolean result = false;

            try {
                result = m_dq.next();
            } catch (Exception e) {
                s_log.error(e);
            }

            return result;
        }

        public String getKey() {
            try {
                return ((BigDecimal)m_dq.get("userID")).toString();
            }  catch (Exception e) {
                s_log.error(e);
            }
            return null;
        }

        public Object getElement() {
            String[] result = null;

            try {
                String id = ((BigDecimal)m_dq.get("userID")).toString();
                String email = (String)m_dq.get("primaryEmail");
                String given = (String)m_dq.get("firstName");
                String family = (String)m_dq.get("lastName");

                result = new String[] {id, email, given, family};
            } catch (Exception e) {
                s_log.error("getElement returning null", e);
            }
            return result;
        }
    }

    /**
     * Takes a 4-tuple from a UserListModel and renders it into
     * a domain-specific XML element
     * <pre>
     *   &lt;demo:user email="asdf@asdf.asdf" first-name="First"
     *        last-name="last"/>
     * </pre>
     */
    private class UserListCellRenderer implements ListCellRenderer {
        public Component getComponent(List list, PageState state,
                                      Object value, String key, int idx,
                                      boolean isSelected) {
            // hidden dependency on UserListModel!
            final String[] tuple = (String[])value;

            return new SimpleComponent() {
                    public void generateXML(PageState ps, Element parent) {
                        javax.xml.parsers.DocumentBuilderFactory dbf =
                            javax.xml.parsers.DocumentBuilderFactory.newInstance();
                        dbf.setNamespaceAware(true);
                        try {
                            javax.xml.parsers.DocumentBuilder db =
                                dbf.newDocumentBuilder();
                            org.w3c.dom.Document domDoc =
                                db.parse(new org.xml.sax.InputSource
                                         (new java.io.StringReader(getXMLString())));
                            com.arsdigita.xml.Document doc =
                                new com.arsdigita.xml.Document(domDoc);
                            parent.addContent(doc.getRootElement());
                        } catch (Exception e) {
                            throw new com.arsdigita.util.UncheckedWrapperException(e);
                        }
                        /*
                          final String namespaceURI =
                          "http://www.arsdigita.com/demo/1.0";
                          Element user = new Element("demo:user", namespaceURI);
                          user.addAttribute("email", tuple[1]);
                          user.addAttribute("first-name", tuple[2]);
                          user.addAttribute("last-name", tuple[3]);
                          parent.addContent(user);
                        */
                    }

                    public String getXMLString() {
                        return "<demo:user xmlns:demo=\"http://www.arsdigita.com/demo/1.0\" email=\"" + tuple[1] + "\" first-name=\"" + tuple[2] + "\" last-name=\"" + tuple[3] + "\"/>";
                    }
                };
        }
    }
}
