/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop;


import java.util.Iterator;


import com.arsdigita.xml.Element;

import com.arsdigita.bebop.form.Hidden;

// This interface contains the XML element name of this class
// in a constant which is used when generating XML
import com.arsdigita.bebop.util.BebopConstants;

/**
 * A container that prints its components in one row, either horizontally or
 * vertically.
 *
 * @author David Lutterkort 
 *
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/BoxPanel.java#8 $
 * */
public class BoxPanel extends SimpleContainer
    implements BebopConstants
{

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/BoxPanel.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * Specifies that components should be laid out left to right.
     */
    public final static int HORIZONTAL = 1;

    /**
     * Specifies that components should be laid out top to bottom.
     */
    public final static int VERTICAL = 2;

    private static final String WIDTH_ATTR = "width";
    private static final String BORDER_ATTR = "border";

    private int m_axis;
    private boolean m_centering;

    // Instance methods

    /**
     * Creates a box panel that lays out its components from top to bottom.
     * The components are not centered.
     */
    public BoxPanel() {
        this(VERTICAL);
    }

    /**
     * Creates a box panel that lays out its components in the given
     * direction. The components are not centered.
     *
     * @param axis the axis to use to lay out the components
     */
    public BoxPanel(int axis) {
        this(axis, false);
    }

    /**
     * Creates a box panel that lays out its components in the given
     * direction and centers them if that is specified.
     *
     * @param axis the axis to use to lay out the components
     * @param centering <code>true</code> if the layout should be centered
     */
    public BoxPanel(int axis, boolean centering) {
        m_axis = axis;
        m_centering = centering;
    }

    /**
     * Adds nodes for the panel and its child components to be rendered,
     * usually in a table. Any hidden widgets directly contained in the box
     * panel are added directly to <code>parent</code> and are not in any
     * of the cells that the box panel generates.
     *
     * <p>Generates DOM fragment:
     * <p><code>&lt;bebop:boxPanel [width=...] border=... center... axis...>
     *   &lt;bebop:cell> cell contents &lt;/bebop:cell>
     * &lt;/bebop:boxPanel></code>
     */
    public void generateXML(PageState state, Element parent) {
        if ( isVisible(state) ) {
            Element panel = parent.newChildElement(BEBOP_BOXPANEL, BEBOP_XML_NS);
            // or: rowPanel/columPanel?
            panel.addAttribute("center", String.valueOf(m_centering));
            panel.addAttribute("axis", String.valueOf(m_axis));
            exportAttributes(panel);

            for (Iterator i = children(); i.hasNext(); ) {
                Component c = (Component) i.next();

                if ( c.isVisible(state) ) {
                    if ( c instanceof Hidden ) {
                        c.generateXML(state, parent);
                    } else {
                        Element cell = panel.newChildElement(BEBOP_CELL, BEBOP_XML_NS);
                        c.generateXML(state, cell);
                    }
                }
            }
        }
    }

    /**
     * Sets the width attribute of the box panel. The given width should be in
     * a form that is legal as the <code>width</code> attribute of an HTML
     * <code>table</code> element.
     *
     * @param w the width of the box panel
     */
    public void setWidth(String w) {
        setAttribute(WIDTH_ATTR, w);
    }

    /**
     * Sets whether a border should be drawn.
     * 
     * @param isBorder <code>true</code> if a border should be drawn
     * @deprecated Use {@link #setBorder(int border)} instead.
     */
    public void setBorder(boolean isBorder) {
        if (isBorder) {
            setAttribute(BORDER_ATTR, "1");
        } else {
            setAttribute(BORDER_ATTR, "0");
        }
    }

    /**
     * 
     * Sets the width of the border to draw around the components. This value
     * will be used for the <code>border</code> attribute in an HTML
     * <code>table</code> element.
     *
     * @param border the width of the border
     */
    public void setBorder(int border) {
        setAttribute(BORDER_ATTR, String.valueOf(border));
    }

}
