/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.mimetypes;

import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.Session;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.mimetypes.converters.ConvertFormat;
import com.arsdigita.mimetypes.converters.PreConvertHTML;
import com.arsdigita.mimetypes.converters.PostConvertHTML;
import com.arsdigita.mimetypes.MimeTypeInitializer;

import com.arsdigita.tools.junit.framework.BaseTestCase;
import java.math.BigDecimal;
import java.lang.Process;
import java.lang.Runtime;
import java.io.*;

import com.arsdigita.util.StringUtils;
import com.arsdigita.db.DbHelper;


/**
 * Test interMedia INSO filter converting of documents
 * to html.
 *
 * @author Jeff Teeters (teeters@arsdigita.com)
 * @version $Revision: #9 $ $Date: 2004/04/07 $
 */

public class ConvertHTMLTest extends BaseTestCase {

    public static final String versionId = "$Id: //core-platform/dev/test/src/com/arsdigita/mimetypes/ConvertHTMLTest.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    // The last test in this file is only run if
    // MimeTypeInitializer.INSO_filter_works is not 0.
    // That test requires a working interMedia INSO Filter.
    // 8.1.6 should pass, 8.1.7 may fail.

    private Session m_session;

    public ConvertHTMLTest(String name) {
        super(name);
    }

    /**
     * Test creating and writing into a PreConvertHTML object
     */
    public void testPreConvertHTML() {
        // Doesn't run under Postgres
        if (DbHelper.getDatabase() == DbHelper.DB_POSTGRES) {
            return;
        }

        final String TEST_CONTENT = "Test pre-convert Content.";
        final BigDecimal EXP_ID = new BigDecimal(237);  // random id

        PreConvertHTML p, p1;

        BigDecimal pID;

        // Create a new PreConvertHTML object
        try {
            p = new PreConvertHTML();
            p.setId(EXP_ID);
            p.setContent(TEST_CONTENT.getBytes());
            pID = EXP_ID;
            p.save();

            p1 = new PreConvertHTML
                (new OID(PreConvertHTML.BASE_DATA_OBJECT_TYPE, pID));
            String foundContent = new String(p1.getContent());
            assertTrue("Content, expected=" + TEST_CONTENT + " found=" +
                       foundContent, TEST_CONTENT.equals(foundContent));

        } catch (DataObjectNotFoundException e) {
            fail(e.getMessage());
            return;
        }

        //Delete object
        try {
            p.delete();

        } catch (Exception e) {
            fail(e.getMessage());
            return;
        }
        try {
            p1 = new PreConvertHTML
                (new OID(PreConvertHTML.BASE_DATA_OBJECT_TYPE, pID));

            //the above line should generate an exception
            fail("PreConvertHTML should have been deleted but still exists.");
        } catch (DataObjectNotFoundException e) {
            //good
        }
    }


    /**
     * Test creating and writing into a PostConvertHTML object
     */
    public void testPostConvertHTML() {
        final String TEST_CONTENT = "Test post-convert Content.";
        final BigDecimal EXP_ID = new BigDecimal(128);  // random id

        PostConvertHTML p, p1;

        BigDecimal pID;

        // Create a new PostConvertHTML object
        try {
            p = new PostConvertHTML();
            p.setId(EXP_ID);
            p.setContent(TEST_CONTENT);
            pID = EXP_ID;
            p.save();

            p1 = new PostConvertHTML
                (new OID(PostConvertHTML.BASE_DATA_OBJECT_TYPE, pID));
            String foundContent = p1.getContent();
            assertTrue("Content, expected=" + TEST_CONTENT + " found=" +
                       foundContent, TEST_CONTENT.equals(foundContent));

        } catch (DataObjectNotFoundException e) {
            fail(e.getMessage());
            return;
        }

        //Delete object
        try {
            p.delete();

        } catch (Exception e) {
            fail(e.getMessage());
            return;
        }
        try {
            p1 = new PostConvertHTML
                (new OID(PostConvertHTML.BASE_DATA_OBJECT_TYPE, pID));

            //the above line should generate an exception
            fail("PostConvertHTML should have been deleted but still exists.");
        } catch (DataObjectNotFoundException e) {
            //good
        }
    }

    /***
     * Get value of ACS_HOME environment variable
     ***/
    private String getACS_HOME() {
        String value = null;
        try {
            String[] cmd = { "/bin/sh", "-c", "echo $" + "ACS_HOME"};
            Process p = Runtime.getRuntime().exec(cmd);
            InputStreamReader isr = new InputStreamReader(p.getInputStream());
            BufferedReader br = new BufferedReader(isr);
            value = br.readLine();
        } catch (Exception e) {
            fail("Unable to get ACS_HOME environment variable.");
        }
        return value;
    }

    /***
     * Load a file from the cms/test/src/convertHTML directory
     ***/
    private byte [] loadFile(String fileName) throws IOException {


        String pathStub = System.getProperty("test.webapp.dir");
        String fullPath = pathStub + "/tests/com/arsdigita/mimetypes/convertHTML/" + fileName;

        byte [] file_bytes = null;
        InputStream fs = new FileInputStream(fullPath);
        file_bytes = new byte[fs.available()];
        fs.read(file_bytes);
        fs.close();
        return file_bytes;
    }


    /**
     * Test conversion of RTF document to html
     */

    public void testRtfConversion() throws Exception {
        // Don't do test if mime-type initializer not setup
        if (MimeTypeStatus.getMimeTypeStatus().getInsoFilterWorks().equals(new BigDecimal(0))) {
            return;
        }
        // First test stripWhiteSpace function
        String in = " <   H>   e \t\n ll/>   o  . \n   ";
        String expected_out = "< H> e ll/> o .";
        String actual_out = StringUtils.stripWhiteSpace(in);
        assertTrue("stripWhiteSpace failed.  Expected = '" +
                   expected_out + "', Found = '" + actual_out + "'",
                   actual_out.equals(expected_out));

        byte [] rtf_in = loadFile("rtftest.rtf");
        expected_out = new String(loadFile("rtftest.html"));
        BigDecimal id = new BigDecimal(1);
        actual_out = ConvertFormat.toHTML(rtf_in);
        if (actual_out == null) {
            fail("Unable to convert rtf document to html.");
        }
        // remove white space before doing matching.  In case version
        // of INSO filter changes.
        expected_out = StringUtils.stripWhiteSpace(expected_out);
        actual_out = StringUtils.stripWhiteSpace(actual_out);
        assertTrue("RTF document converted, but result does not " +
                   "match expected.  EXPECTED=" + expected_out + " FOUND=" +
                   actual_out, expected_out.equals(actual_out));
    }
}
