/*
 * Copyright (C) 2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.categorization;

import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.DataQuery;

import org.apache.log4j.Logger;

import java.math.BigDecimal;

public class AncestorsTest extends CategoryTestCase {
    private final static Logger s_log = Logger.getLogger(AncestorsTest.class);

    private Category m_parent;
    private Category m_child;
    private Category m_toddler;
    private BigDecimal m_parentID;
    private BigDecimal m_childID;
    private BigDecimal m_toddlerID;

    public AncestorsTest(String name) {
        super(name);
    }

    public void setUp() {
        m_parent = new Category();
        m_parent.setName("Parent");
        m_parentID = m_parent.getID();

        m_child = new Category();
        m_child.setName("Child");
        m_childID = m_child.getID();

        m_toddler = new Category();
        m_toddler.setName("M_Toddler");
        m_toddlerID = m_toddler.getID();
    }

    public void testAncestors() {
        m_parent.save();
        m_child.save();
        m_toddler.save();

        validatePath(m_parentID, new BigDecimal[] { m_parentID });
        validatePath(m_childID, new BigDecimal[] { m_childID });
        validatePath(m_toddlerID, new BigDecimal[] { m_toddlerID });

        m_child.addChild(m_toddler);
        m_toddler.setDefaultParentCategory(m_child);
        m_child.save();
        m_toddler.save();

        validatePath(m_parentID, new BigDecimal[] { m_parentID });
        validatePath(m_childID, new BigDecimal[] { m_childID });
        validatePath(m_toddlerID, new BigDecimal[] { m_childID, m_toddlerID });

        m_parent.addChild(m_child);
        m_parent.save();
        m_child.save();

        validatePath(m_parentID, new BigDecimal[] { m_parentID });
        validatePath(m_childID, new BigDecimal[] { m_childID });
        validatePath(m_toddlerID, new BigDecimal[] { m_childID, m_toddlerID });

        m_child.setDefaultParentCategory(m_parent);
        m_parent.save();
        m_child.save();

        validatePath(m_parentID, new BigDecimal[] { m_parentID });
        validatePath(m_childID, new BigDecimal[] { m_parentID, m_childID });
        validatePath(m_toddlerID, new BigDecimal[] { m_parentID, m_childID, m_toddlerID });

        m_parent.removeChild(m_child);
        m_parent.save();
        m_child.save();

        validatePath(m_parentID, new BigDecimal[] { m_parentID });
        validatePath(m_childID, new BigDecimal[] { m_childID });
        validatePath(m_toddlerID, new BigDecimal[] { m_childID, m_toddlerID });

        m_child.removeChild(m_toddler);
        m_child.save();
        m_toddler.save();
        
        validatePath(m_parentID, new BigDecimal[] { m_parentID });
        validatePath(m_childID, new BigDecimal[] { m_childID });
        validatePath(m_toddlerID, new BigDecimal[] { m_toddlerID });
    }


    private void validatePath(BigDecimal id,
                              BigDecimal[] path) {
        // Messed up part Java / part SQL denormalizatio
        // means we can't rely on the Category dataobject
        // having correct value, so fetch direct from DB.
        DataQuery anc = SessionManager.getSession().retrieveQuery("com.arsdigita.categorization.fetchAncestors");
        anc.setParameter("id", id);
        
        assertTrue(anc.next());
        
        String actual = (String)anc.get("ancestors");
        anc.close();
        
        StringBuffer expected = new StringBuffer();
        for (int i = 0 ; i < path.length ; i++) {
            expected.append(path[i] + "/");
        }
        
        s_log.debug("Compare " + expected + " to " + actual);

        assertEquals(expected.toString(),
                     actual);
    }
}
