/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.web;

import com.arsdigita.util.Assert;
import org.apache.log4j.Logger;

/**
 * <p>A signal that requests to commit or abort the current transaction
 * and to send a redirect to a new URL.  BaseServlet traps this signal
 * when it is thrown and finishes the transaction before it sends the
 * redirect to the response.  This way the client cannot see state
 * inconsistent with work performed in the previous request.</p>
 *
 * <p><code>RedirectSignal</code>s are usually sent after doing work
 * on behalf of the user:</p>
 *
 * <blockquote><pre>
 * private final void saveUserSettings(final HttpServletRequest sreq) {
 *     m_user.setGivenName("Gibbon");
 *     m_user.setFamilyName("Homily");
 *
 *     m_user.save();
 *
 *     // The boolean argument true signifies that we want to commit
 *     // the transaction.
 *     throw new RedirectSignal(URL.here(sreq, "/user-detail.jsp"), true);
 * }
 * </pre></blockquote>
 *
 * @see com.arsdigita.web.BaseServlet
 * @see com.arsdigita.web.LoginSignal
 * @see com.arsdigita.web.ReturnSignal
 * @author Justin Ross &lt;jross@redhat.com&gt;
 * @version $Id: //core-platform/dev/src/com/arsdigita/web/RedirectSignal.java#8 $
 */
public class RedirectSignal extends TransactionSignal {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/web/RedirectSignal.java#8 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger(RedirectSignal.class);

    private final String m_url;

    public RedirectSignal(final String url, final boolean isCommitRequested) {
        super(isCommitRequested);

        if (Assert.isAssertEnabled()) {
            Assert.assertNotNull(url, "String url");
            Assert.assertTrue(url.startsWith("http") || url.startsWith("/"),
                              "The URL is relative and won't dispatch " +
                              "correctly under some servlet containers; " +
                              "the URL is '" + url + "'");
        }

        if (s_log.isDebugEnabled()) {
            s_log.debug("Request for redirect to URL '" + url + "'",
                        new Throwable());
        }

        m_url = url;
    }

    public RedirectSignal(final URL url, final boolean isCommitRequested) {
        this(url.toString(), isCommitRequested);
    }

    public final String getDestinationURL() {
        return m_url;
    }
}
