/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.web;

import com.arsdigita.dispatcher.DispatcherHelper;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletConfig;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;


/**
 * Allow arbitrary prefixes to be set up in web.xml such as /textonly,
 * /printer, /debug, etc. This enables the entire site to be switched
 * to alternative modes of operation e.g., completely switching to an
 * alternative set of stylesheets. For example,
 *
 * <pre>
 * &lt;servlet&gt;
 *   &lt;servlet-name&gt;TextOnlyServlet&lt;/servlet-name&gt;
 *   &lt;display-name&gt;Text Only Servlet&lt;/display-name&gt;
 *   &lt;servlet-class&gt;
 *    com.arsdigita.web.InternalPrefixerServlet
 *   &lt;/servlet-class&gt;
 *   &lt;init-param&gt;
 *    &lt;param-name&gt;prefix&lt;/param-name&gt;
 *    &lt;param-value&gt;/text&lt;/param-value&gt;
 *  &lt;/init-param&gt;
 * &lt;/servlet&gt;
 *
 * &lt;servlet-mapping&gt;
 *    &lt;servlet-name&gt;TextOnlyServlet&lt;/servlet-name&gt;
 *    &lt;url-pattern&gt;/textonly/*&lt;/url-pattern&gt;
 * &lt;/servlet-mapping&gt;
 * </pre>
 *
 * The above entry, in conjunction with a set of stylesheets
 * that producedtext only output, would enable a text only
 * part of the site with a /textonly/ URL prefix.
 *
 */
public class InternalPrefixerServlet extends HttpServlet {

    private static final Logger s_log =
        Logger.getLogger(InternalPrefixerServlet.class);

    private String m_prefix;

    public void init()
        throws ServletException {
        ServletConfig conf = getServletConfig();

        m_prefix = (String)conf.getInitParameter("prefix");

        if (s_log.isDebugEnabled()) {
            s_log.debug("Prefix is " + m_prefix);
        }
    }

    protected void service(HttpServletRequest req,
                           HttpServletResponse resp)
        throws ServletException,
               java.io.IOException {

        String path = req.getPathInfo();
        if (path == null || path.trim().length() == 0) {
            // if path is null or the empty string then trying
            // to get a RequestDispatcher will result in an NPE
            path = "/";
        }
        if (s_log.isDebugEnabled()) {
            s_log.debug("Forwarding " + path);
        }

        DispatcherHelper.setDispatcherPrefix(req, m_prefix);

        ServletContext context = getServletContext();
        RequestDispatcher rd = context.getRequestDispatcher(path);
        rd.forward(req, resp);
    }

}
