/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.ui.sitemap;


import com.arsdigita.ui.util.GlobalizationUtil ;

import com.arsdigita.bebop.BebopMapDispatcher;
import com.arsdigita.bebop.SplitPanel;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.BoxPanel;
import com.arsdigita.bebop.Page;
import com.arsdigita.bebop.PageFactory;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.dispatcher.DispatcherHelper;
import com.arsdigita.dispatcher.RequestContext;
import com.arsdigita.web.RedirectSignal;
import com.arsdigita.web.URL;
import com.arsdigita.web.Web;
import com.arsdigita.web.LoginSignal;
import com.arsdigita.kernel.SiteNode;
import com.arsdigita.kernel.User;
import com.arsdigita.kernel.permissions.PermissionDescriptor;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.kernel.security.UserContext;
import com.arsdigita.globalization.GlobalizedMessage;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.log4j.Logger;

/**
 * Dispatcher for SiteMap Admin functionality
 */
public class SiteMapDispatcher extends BebopMapDispatcher {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/ui/sitemap/SiteMapDispatcher.java#15 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log =
        Logger.getLogger(SiteMapDispatcher.class);

    static final String SEARCH_KEY = "search";

    /**
     * Constructor.  Instantiates the subsite url/page mapping.
     */
    public SiteMapDispatcher() {
        super();

        s_log.debug("Construct SiteMapDispatcher");

        Map  m = new HashMap();
        m.put("", buildAdminPage());
        m.put("denied", buildDeniedPage());
        setMap(m);
    }

    public void dispatch(HttpServletRequest req,
                         HttpServletResponse resp,
                         RequestContext ctx)
            throws IOException, javax.servlet.ServletException {
        /**
         * Authenicate user.
         */
        String requestURI = Web.getContext().getRequestURL().getRequestURI();

        if (requestURI.endsWith("/denied")) {
            super.dispatch(req, resp, ctx);
            return;
        }

        UserContext userCtx = Web.getUserContext();

        // If the user is not logged in, then redirect to the log in page.

        if (!userCtx.isLoggedIn()) {
            throw new LoginSignal(req);
        } else {
            // Check if user has an admin privilege.
            String processedURL = ctx.getProcessedURLPart();

            SiteNode node;

            node = SiteNode.getSiteNode(processedURL);

            User user = userCtx.getUser();

            if (user == null) {
                throw new RuntimeException("User does not exist");
            }

            PermissionDescriptor admin = new PermissionDescriptor
                (PrivilegeDescriptor.ADMIN, node, user);

            // Turn off caching for everything...
            DispatcherHelper.cacheDisable(resp);

            if (PermissionService.checkPermission(admin)) {
                super.dispatch(req, resp, ctx);
            } else {
                final URL url = URL.there(req, node.getURL() + "denied");

                throw new RedirectSignal(url, false);
            }
        }
    }

    /**
     * "Access Denied" page for the SiteMap.
     */

    private Page buildDeniedPage() {
        Page p = PageFactory.buildPage("admin", new Label(new GlobalizedMessage
                                                          ("ui.admin.dispatcher.accessDenied",
                                                           "com.arsdigita.ui.admin.AdminResources" )));

        Label label = new Label(GlobalizationUtil.globalize("ui.sitemap.access_denied_to_sitemap"));
        label.setClassAttr("AccessDenied");
        p.add(label);

        p.lock();

        return p;
    }

    private Page buildAdminPage() {
        Page p = PageFactory.buildPage("admin", "Sitemap Administration");

        SiteListing listing = new SiteListing();
        listing.setClassAttr("navbar");

        SingleSelectionModel m = listing.getTree().getSelectionModel();
        SiteMapAdminPane details = new SiteMapAdminPane(m, listing.getCFGLink());

        BoxPanel box = new BoxPanel();
        box.setClassAttr("main");
        box.add(details);

        SplitPanel panel = new SplitPanel();
        panel.setClassAttr("sidebarNavPanel");
        panel.setLeftComponent(listing);
        panel.setRightComponent(box);

        p.add(panel);
        p.lock();

        return p;
    }

}
