/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.ui.login;

import com.arsdigita.bebop.event.ParameterEvent;
import com.arsdigita.bebop.parameters.NotNullValidationListener;
import com.arsdigita.bebop.parameters.ParameterData;

/**
 * A NotNullValidationListener that checks whether the string value of a
 * parameter is a strong password.
 *
 * @author Sameer Ajmani
 */

public class PasswordValidationListener
    extends NotNullValidationListener
{
    /**
     * Minimum length for passwords.
     */
    public static final int MIN_LENGTH = 6;

    /**
     * Minimum number of character types for passwords.  Valid values are on
     * range [1,4].
     */
    public static final int MIN_TYPES = 1;

    public PasswordValidationListener() {
        super();
    }

    public PasswordValidationListener(String label) {
        super(label);
    }

    /**
     * Checks whether the string value of the parameter meets minimum-length
     * and composition requirements for strong passwords.
     */
    public void validate(ParameterEvent e) {
        super.validate(e);

        ParameterData data = e.getParameterData();
        if (!data.isValid()) {
            // already has errors from superclass
            return;
        }

        // check length
        String password = data.getValue().toString();
        if (password.length() < MIN_LENGTH) {
            data.addError(LoginHelper.getMessage
                          ("login.passwordValidation.minLengthError",
                           new Object[] { new Integer(MIN_LENGTH) }));
            return;
        }

        // check for whitespace
        if (contains(password, whites)) {
            data.addError(LoginHelper.getMessage
                          ("login.passwordValidation.whitespaceError"));
            return;
        }

        // check character mix
        int charSets = 0;
        if (contains(password, uppers)) {
            charSets++;
        }
        if (contains(password, lowers)) {
            charSets++;
        }
        if (contains(password, digits)) {
            charSets++;
        }
        if (contains(password, others)) {
            charSets++;
        }
        if (charSets < MIN_TYPES) {
            data.addError(LoginHelper.getMessage
                          ("login.passwordValidation.minTypesError",
                           new Object[] { new Integer(MIN_TYPES) }));
            return;
        }
    }
    private static interface CharSet {
        public boolean contains(char c);
    }
    private static boolean contains(String s, CharSet set) {
        for (int i = 0; i < s.length(); i++) {
            if (set.contains(s.charAt(i)))
                return true;
        }
        return false;
    }
    private static CharSet whites = new CharSet() {
            public boolean contains(char c) {
                return Character.isWhitespace(c);
            }
        };
    private static CharSet uppers = new CharSet() {
            public boolean contains(char c) {
                return (Character.isLetter(c)
                        && Character.isUpperCase(c));
            }
        };
    private static CharSet lowers = new CharSet() {
            public boolean contains(char c) {
                return (Character.isLetter(c)
                        && Character.isLowerCase(c));
            }
        };
    private static CharSet digits = new CharSet() {
            public boolean contains(char c) {
                return Character.isDigit(c);
            }
        };
    private static CharSet others = new CharSet() {
            public boolean contains(char c) {
                return !(Character.isLetterOrDigit(c)
                         || Character.isISOControl(c)
                         || Character.isWhitespace(c));
            }
        };
}
