/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.toolbox.ui;



import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.CompoundComponent;
import com.arsdigita.bebop.SimpleContainer;
import com.arsdigita.bebop.RequestLocal;
import com.arsdigita.bebop.PageState;


import com.arsdigita.kernel.ACSObject;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.Party;

import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.kernel.permissions.PermissionDescriptor;
import com.arsdigita.kernel.permissions.PermissionService;

import com.arsdigita.xml.Element;


/**
 * <p>This component wraps two other components and chooses one of
 * them based on whether or not a given party (specified in a
 * RequestLocal) has a given privilege on a given object (also
 * specified in a RequestLocal).</p>
 **/
public class PrivilegedComponentSelector extends CompoundComponent {

    private PrivilegeDescriptor m_requiredPrivilege;
    private RequestLocal m_objectRL;

    private Component m_privilegedComponent;
    private Component m_unprivilegedComponent;


    /**
     * <p>Construct a new PrivilegedComponentSelector.</p>
     *
     * @param requiredPrivilege Privilege that will be checked
     * @param objectRL RequestLocal in which the object on which
     * permission will be checked is stored
     * @param privilegedComponent Component to be displayed when the
     * current user has the required privilege on the specified object
     * @param unprivilegedComponent Component to be displayed when the
     * current user lacks the required privilege on the specified
     * object
     **/
    public PrivilegedComponentSelector(PrivilegeDescriptor requiredPrivilege,
                                       RequestLocal objectRL,
                                       Component priviledgedComponent,
                                       Component unprivilegedComponent) {
        super(new SimpleContainer());
        getContainer().add(priviledgedComponent);
        getContainer().add(unprivilegedComponent);

        m_requiredPrivilege = requiredPrivilege;
        m_objectRL = objectRL;

        m_privilegedComponent = priviledgedComponent;
        m_unprivilegedComponent = unprivilegedComponent;
    }

    public void generateXML(PageState ps, Element parent) {
        ACSObject object = (ACSObject)m_objectRL.get(ps);
        Party party  = Kernel.getContext().getParty();

        if ((object == null) || (party == null)) {
            m_privilegedComponent.setVisible(ps, false);
            m_unprivilegedComponent.setVisible(ps, true);
        } else {
            PermissionDescriptor perm =
                new PermissionDescriptor(m_requiredPrivilege, object, party);
            if (PermissionService.checkPermission(perm)) {
                m_privilegedComponent.setVisible(ps, true);
                m_unprivilegedComponent.setVisible(ps, false);
            } else {
                m_privilegedComponent.setVisible(ps, false);
                m_unprivilegedComponent.setVisible(ps, true);
            }
        }

        super.generateXML(ps, parent);
    }
}
