/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search.converter;

import com.arsdigita.mimetypes.MimeType;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileInputStream;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.IOException;
import org.apache.poi.poifs.filesystem.*;
import org.apache.poi.hssf.record.*;
import org.apache.poi.hssf.eventusermodel.*;

import org.apache.log4j.Logger;

/**
 * This class provides the mechanism to perform a conversion from
 * Excel to standard text
 */
public class ExcelConverter extends BaseConverter {
    

    /**
     *  Returns a string array representing all of the files types
     *  used by the given converter
     */
    protected String[] getFileExtensions() {
        String[] extensions = {"xls"};
        return extensions;
    }

    /**
     *  This takes in a document represented as an InputStream and returns
     *  a text representation of that document.
     */
    public String convertDocument(InputStream stream) throws ConversionException {
        try {
            // create a new org.apache.poi.poifs.filesystem.Filesystem
            POIFSFileSystem poifs = new POIFSFileSystem(stream);
            // get the Workbook (excel part) stream in a InputStream
            InputStream din = poifs.createDocumentInputStream("Workbook");
            // construct out HSSFRequest object
            HSSFRequest req = new HSSFRequest();
            // lazy listen for ALL records with the listener shown above
            StringBuffer output = new StringBuffer();
            req.addListenerForAllRecords(new ExcelEvent(output));
            // create our event factory
            HSSFEventFactory factory = new HSSFEventFactory();
            // process our events based on the document input stream
            factory.processEvents(req, din);
            // once all the events are processed close our file input stream
            stream.close();
            // and our document input stream (don't want to leak these!)
            din.close();
            return output.toString();
        } catch (IOException ioe) {
            throw new ConversionException(ioe);
        }
    }


    /**
     * This example shows how to use the event API for reading a file.
     */
    public class ExcelEvent implements HSSFListener {
        private SSTRecord sstrec;
        private Logger s_log = Logger.getLogger(MimeType.class);
        
        private StringBuffer m_output;

        public ExcelEvent(StringBuffer output) {
            m_output = output;
        }

        /**
         * This method listens for incoming records and handles 
         *  them as required.
         *
         *  A lot of this is taken from the POI example
         *
         * @param record    The record that was found while reading.
         */
        public void processRecord(Record record) {
            switch (record.getSid()) {
            case BOFRecord.sid:
                if (s_log.isDebugEnabled()) {
                    BOFRecord bof = (BOFRecord) record;
                    if (bof.getType() == bof.TYPE_WORKBOOK) {
                        s_log.debug("Encountered workbook");
                        // assigned to the class level member
                    } else if (bof.getType() == bof.TYPE_WORKSHEET) {
                        s_log.debug("Encountered sheet reference");
                    }
                }
                break;
            case NumberRecord.sid:
                NumberRecord numrec = (NumberRecord) record;
                m_output.append(" " + numrec.getValue());
                break;
            case SSTRecord.sid:
                sstrec = (SSTRecord) record;
                break;
            case LabelSSTRecord.sid:
                LabelSSTRecord lrec = (LabelSSTRecord) record;
                m_output.append(" " + sstrec.getString(lrec.getSSTIndex()));
                break;
            }
        }
    }
} 
