/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.runtime;

import com.arsdigita.util.jdbc.JDBCURLParameter;
import com.arsdigita.util.parameter.IntegerParameter;
import com.arsdigita.util.parameter.Parameter;
import org.apache.log4j.Logger;

/**
 * A configuration record for configuration of the runtime
 * environment.
 *
 * @author Justin Ross &lt;jross@redhat.com&gt;
 * @version $Id: //core-platform/dev/src/com/arsdigita/runtime/RuntimeConfig.java#8 $
 */
public final class RuntimeConfig extends AbstractConfig {
    public final static String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/runtime/RuntimeConfig.java#8 $" +
        "$Author: mbooth $" +
        "$DateTime: 2004/05/28 09:51:17 $";

    private static final Logger s_log = Logger.getLogger(RuntimeConfig.class);

    private static RuntimeConfig s_config;

    /**
     * Returns the singleton configuration record for the runtime
     * environment.
     *
     * @return The <code>RuntimeConfig</code> record; it cannot be null
     */
    public static final synchronized RuntimeConfig getConfig() {
        if (s_config == null) {
            s_config = new RuntimeConfig();
            s_config.require("ccm-core/runtime.properties");
        }

        return s_config;
    }

    private final Parameter m_url;
    private final Parameter m_poolSize;
    private final Parameter m_pingInterval;
    private final Parameter m_queryCacheSize;

    /**
     * Constructs an empty RuntimeConfig object.
     **/

    public RuntimeConfig() {
        m_url = new JDBCURLParameter("waf.runtime.jdbc_url");
        m_poolSize = new IntegerParameter
            ("waf.runtime.jdbc_pool_size", Parameter.OPTIONAL,
             new Integer(10));
        m_pingInterval = new IntegerParameter
            ("waf.runtime.jdbc_ping_interval", Parameter.OPTIONAL,
             new Integer(30000));
        m_queryCacheSize = new IntegerParameter
            ("waf.runtime.query_cache_size", Parameter.OPTIONAL,
             new Integer(2000));

        register(m_url);
        register(m_poolSize);
        register(m_pingInterval);
        register(m_queryCacheSize);

        loadInfo();
    }

    /**
     * Returns the default JDBC URL for the current runtime.
     *
     * @return A <code>String</code> JDBC URL; it cannot be null
     */
    public final String getJDBCURL() {
        return (String) get(m_url);
    }

    /**
     * Returns the maximum size to be used for the connection pool.
     *
     * @return An integer limit on the number of JDBC connections
     * allowed open at once.
     */

    public final int getJDBCPoolSize() {
        return ((Integer) get(m_poolSize)).intValue();
    }

    public final long getJDBCPingInterval() {
        return ((Integer) get(m_pingInterval)).longValue();
    }

    public final int getQueryCacheSize() {
        return ((Integer) get(m_queryCacheSize)).intValue();
    }

}
