/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.portal;

import com.arsdigita.kernel.ResourceType;
import com.arsdigita.web.ApplicationType;
import com.arsdigita.kernel.PackageType;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.util.Assert;
import org.apache.log4j.Logger;
import java.math.BigDecimal;


/**
 * 
 *
 * @author Justin Ross
 */
public class PortletType extends ResourceType {

    private static final Logger s_cat =
        Logger.getLogger(PortletType.class.getName());

    // PortletType uses the data object type of ApplicationType.
    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.portal.PortletType";


    protected String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    public static final String WIDE_PROFILE = "wide";
    public static final String NARROW_PROFILE = "narrow";

    protected PortletType(DataObject dataObject) {
        super(dataObject);
    }

    // Create from packageType.
    protected PortletType
        (String dataObjectType, String title, String profile,
         String portletObjectType) {
        super(dataObjectType);

        Assert.assertNotNull(title, "title");
        Assert.assertNotNull(profile, "profile");
        Assert.assertNotNull(portletObjectType, "portletObjectType");

        PackageType packageType = null;

        try {
            packageType = PackageType.findByKey("portal");
        } catch (DataObjectNotFoundException nfe) {
            String message =
                "The PackageType 'portal' is not installed.  It must be " +
                "installed in order to create a new PortletType.";

            s_cat.error(message);

            throw new IllegalStateException(message);
        }

        setTitle(title);
        setResourceObjectType(portletObjectType);

        // Defaults for portlets.
        setProfile(profile);
        setFullPageView(false);
        setEmbeddedView(true);
    }

    public static PortletType createPortletType
        (String title, String profile, String portletObjectType) {
        return new PortletType
            (BASE_DATA_OBJECT_TYPE, title, profile, portletObjectType);
    }

    public static PortletType retrievePortletType(BigDecimal id) {
        Assert.assertNotNull(id, "id");

        return PortletType.retrievePortletType
            (new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    public static PortletType retrievePortletType(OID oid) {
        Assert.assertNotNull(oid, "oid");

        DataObject dataObject = SessionManager.getSession().retrieve(oid);

        Assert.assertNotNull(dataObject);

        return PortletType.retrievePortletType(dataObject);
    }

    public static PortletType retrievePortletType(DataObject dataObject) {
        Assert.assertNotNull(dataObject, "dataObject");

        return new PortletType(dataObject);
    }

    public static PortletType retrievePortletTypeForPortlet
        (String portletObjectType) {
        Assert.assertNotNull(portletObjectType, "portletObjectType");

        DataCollection collection =
            SessionManager.getSession().retrieve(BASE_DATA_OBJECT_TYPE);

        Assert.assertNotNull(collection, "collection");

        collection.addEqualsFilter("objectType", portletObjectType);

        PortletType portletType = null;

        if (collection.next()) {
            portletType = PortletType.retrievePortletType
                (collection.getDataObject());
        } else {
            s_cat.warn("No portlet type found that matches \"" +
                       portletObjectType + ".\"  Check that the portlet " +
                       "type is registered in the system.");
        }

        collection.close();
        return portletType;
    }

    public static PortletTypeCollection retrieveAllPortletTypes() {
        DataCollection collection =
            SessionManager.getSession().retrieve(BASE_DATA_OBJECT_TYPE);

        Assert.assertNotNull(collection, "collection");
        collection.addEqualsFilter("hasEmbeddedView", Boolean.TRUE);

        return new PortletTypeCollection(collection);
    }

    //
    // Association properties
    //

    public void setProviderApplicationType(ApplicationType applicationType) {
        Assert.assertNotNull(applicationType, "applicationType");

        setAssociation("providerApplicationType", applicationType);
    }

    public void setProviderApplicationType(String applicationObjectType) {
        ApplicationType applicationType =
            ApplicationType.retrieveApplicationTypeForApplication
            (applicationObjectType);

        setProviderApplicationType(applicationType);
    }

    public ApplicationType getProviderApplicationType() {
        DataObject dobj = (DataObject) get("providerApplicationType");
        if (dobj == null) { return null; }
        return ApplicationType.retrieveApplicationType(dobj);
    }

    public boolean hasFullPageView() {
        Boolean hasFullPageView = (Boolean) get("hasFullPageView");

        Assert.assertNotNull(hasFullPageView, "hasFullPageView");

        return hasFullPageView.booleanValue();
    }

    protected void setFullPageView(boolean hasFullPageView) {
        set("hasFullPageView", new Boolean(hasFullPageView));
    }

    public boolean hasEmbeddedView() {
        Boolean hasEmbeddedView = (Boolean)get("hasEmbeddedView");

        Assert.assertNotNull(hasEmbeddedView, "hasEmbeddedView");

        return hasEmbeddedView.booleanValue();
    }

    protected void setEmbeddedView(boolean hasEmbeddedView) {
        set("hasEmbeddedView", new Boolean(hasEmbeddedView));
    }

    // Can return null.
    public String getProfile() {
        String profile = (String) get("profile");

        return profile;
    }

    // Param profile can be null.
    protected void setProfile(String profile) {
        set("profile", profile);
    }

}
