/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel.security;

import java.util.Map;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginException;
import org.apache.log4j.Logger;

/**
 * Provides methods for saving, retrieving, and deleting a string value from
 * HTTP requests and responses.  Used with
 * <code>CredentialLoginModule</code> to get, set, and delete client
 * authentication credentials.
 *
 * @see CredentialLoginModule
 *
 * @author Sameer Ajmani
 **/
public abstract class CredentialManager {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/kernel/security/CredentialManager.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    private static final Logger s_log =
        Logger.getLogger(URLManager.class.getName());

    // the login module associated with this manager
    private CredentialLoginModule m_module;

    /**
     * Constructs a new <code>CredentialManager</code>.
     **/
    protected CredentialManager() {
    }

    /**
     * Returns the <code>CredentialLoginModule</code> associated with this
     * <code>CredentialManager</code>.
     **/
    protected CredentialLoginModule getModule() {
        return m_module;
    }

    /**
     * Initializes this with the login context information.  Called by
     * <code>CredentialLoginModule.initialize()</code>.
     *
     * @param module the <code>CredentialLoginModule</code> associated with
     * this <code>CredentialManager</code>.
     **/
    protected void initialize(CredentialLoginModule module,
                              Subject subject,
                              CallbackHandler handler,
                              Map shared,
                              Map options) {
        m_module = module;
    }

    /**
     * Retrieves the value of the credential named by
     * <code>getModule().getCredentialName()</code>.
     *
     * @return the credential value, never null.
     *
     * @throws CredentialNotFoundException if credential does not exist.
     *
     * @throws LoginException if an error occurs.
     **/
    protected abstract String getValue()
        throws LoginException;

    /**
     * Determines whether <code>setValue()</code> should be called.
     *
     * @param value the new value for the credential
     *
     * @return true if <code>setValue()</code> should be called
     *
     * @throws LoginException if an error occurs.
     **/
    protected abstract boolean shouldSetValue(String value)
        throws LoginException;

    /**
     * Sets the credential named by
     * <code>getModule().getCredentialName()</code> to the given value.
     *
     * @param value the new value for the credential
     *
     * @throws LoginException if an error occurs.
     **/
    protected abstract void setValue(String value)
        throws LoginException;

    /**
     * Deletes the credential named by
     * <code>getModule().getCredentialName()</code>.
     *
     * @throws LoginException if an error occurs.
     **/
    protected abstract void deleteValue()
        throws LoginException;
}
