/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel;

import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.TransactionContext;
import org.apache.log4j.Logger;

/**
 * Represents a database transaction.
 *
 * @author Justin Ross
 */
public final class DatabaseTransaction {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/kernel/DatabaseTransaction.java#7 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (DatabaseTransaction.class);

    private final TransactionContext m_context;
    private boolean m_isCommitRequested;

    public DatabaseTransaction() {
        m_context = SessionManager.getSession().getTransactionContext();
        m_isCommitRequested = true;
    }

    /**
     * Starts the transaction if it hasn't already been started.
     */
    public final void begin() {
        if (!m_context.inTxn()) {
            s_log.debug("Beginning transaction");
            m_context.beginTxn();
        } else {
            s_log.debug("Tried to begin, but the transaction was already " +
                        "begun");
        }
    }

    /**
     * Finishes the transaction if it hasn't already ended.  Commits
     * or aborts based on the value of {@link #isCommitRequested()}.
     */
    public final void end() {
        if (isCommitRequested()) {
            commit();
        } else {
            abort();
        }
    }

    /**
     * Commits the transaction if it hasn't already ended.
     */
    public final void commit() {
        if (m_context.inTxn()) {
            s_log.debug("Committing transaction");
            m_context.commitTxn();
        } else {
            s_log.debug("Tried to commit, but the transaction was already " +
                        "finished");
        }
    }

    /**
     * Aborts the transaction if it hasn't already ended.
     */
    public final void abort() {
        if (m_context.inTxn()) {
            s_log.debug("Aborting transaction");
            m_context.abortTxn();
        } else {
            s_log.debug("Tried to abort, but the transaction was already " +
                        "finished");
        }
    }

    /**
     * Tells whether the transaction should be committed or aborted at
     * its end.  Transactions are by default committed.
     */
    public final boolean isCommitRequested() {
        return m_isCommitRequested;
    }

    /**
     * Requests that the transaction be committed or aborted at its
     * end.
     */
    public final void setCommitRequested(final boolean isCommitRequested) {
        m_isCommitRequested = isCommitRequested;
    }

    public String toString() {
        return super.toString() + " " +
            "[" + m_context.inTxn() + "," + isCommitRequested() + "]";
    }
}
