/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.globalization;

import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.db.Sequences;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;
import java.math.BigDecimal;

/**
 * <p>
 * Charset DomainObject implements a persistent Charset object. A charset, or
 * character set, is ...
 * </p>
 *
 * @author Yon Feldman
 * @version $Revision: #10 $ $Date: 2004/04/07 $
 */
public class Charset extends DomainObject {
    public final static String versionId = "$Id: //core-platform/dev/src/com/arsdigita/globalization/Charset.java#10 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    public final static String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.globalization.Charset";

    /**
     * <p>
     * Constructor. Retrieves an instance for a new Charset
     * </p>
     *
     */
    public Charset() {
        super(BASE_DATA_OBJECT_TYPE);
    }

    /**
     * <p>
     * Constructor. Retrieves a persistent Charset object from the database.
     * </p>
     *
     * @param oid Object ID of the Charset to retrieve.
     *
     * @exception DataObjectNotFoundException Thrown if we cannot retrieve a
     *            DataObject for the specified OID.
     */
    public Charset(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }

    /**
     * <p>
     * Constructor. Retrieves a persistent Charset object from the database.
     * </p>
     *
     * @param dataObject DataObject of the Charset to retrieve.
     *
     */
    public Charset(DataObject dataObject) {
        super(dataObject);
    }

    protected void initialize() {
        super.initialize();
        try {
            if (isNew() && getID() == null) {
                setID(Sequences.getNextValue());
            }
        } catch (java.sql.SQLException ex) {
            throw new UncheckedWrapperException("cannot set charset id", ex);
        }
    }

    /**
     * <p>
     * Returns the appropriate object type for a Charset so that the proper
     * type validation can take place when retrieving Charset by OID.
     * </p>
     *
     * @return String The fully qualified name of the base data object type
     *         for the Charset DataObject.
     */
    public String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    /**
     * <p>
     * Return ID.
     * </p>
     *
     * @return BigDecimal The ID of this DataObject
     */
    public BigDecimal getID() {
        return (BigDecimal) get("id");
    }

    /**
     * <p>
     * Set the ID
     * </p>
     *
     * @param id ID
     */
    private void setID(BigDecimal id) {
        if (id == null) {
            throw new IllegalArgumentException("id cannot be empty.");
        }

        if (isNew()) {
            set("id", id);
        }
    }

    /**
     * <p>
     * Returns the charset.
     * </p>
     *
     * @return String charset
     */
    public String getCharset() {
        return (String) get("charset");
    }

    /**
     * <p>
     * Set the charset
     * </p>
     *
     * @param String charset
     */
    public void setCharset(String charset) {
        if (charset == null || charset.length() == 0) {
            throw new IllegalArgumentException("charset cannot be empty.");
        }

        set("charset", charset);
    }

    protected void afterSave() {
        super.afterSave();
        Globalization.loadLocaleToCharsetMap();
    }

    protected void afterDelete() {
        super.afterDelete();
        Globalization.loadLocaleToCharsetMap();
    }
}
