/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.formbuilder;


import com.arsdigita.bebop.parameters.ParameterModel;

import com.arsdigita.formbuilder.util.FormBuilderUtil;


/**
 * Represents the metadata of one attribute of a <code>AttributeMetaDataProvider</code>.
 *
 * @author Peter Marklund
 * @version $Id: //core-platform/dev/src/com/arsdigita/formbuilder/AttributeMetaData.java#7 $
 *
 */
public class AttributeMetaData {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/formbuilder/AttributeMetaData.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private String m_parameterName;
    private String m_label;
    private boolean m_isRequired = false;
    private boolean m_isMultiple = false;
    private AttributeType m_attributeType;


    //*** Constructors

    public AttributeMetaData(String parameterName) {

        this(parameterName, null, false, false, null);
    }

    public AttributeMetaData(String parameterName, boolean isRequired) {

        this(parameterName, null, isRequired, false, null);
    }

    public AttributeMetaData(String parameterName, boolean isRequired, boolean isMultiple) {

        this(parameterName, null, isRequired, isMultiple, null);
    }

    public AttributeMetaData(String parameterName, AttributeType attributeType) {

        this(parameterName, null, false, false, attributeType);
    }

    public AttributeMetaData(String parameterName, String label) {

        this(parameterName, label, false, false, null);
    }

    public AttributeMetaData(String parameterName, String label, boolean isRequired) {

        this(parameterName, label, isRequired, false, null);
    }

    public AttributeMetaData(String parameterName, String label, AttributeType attributeType) {

        this(parameterName, label, false, false, attributeType);
    }

    /**
     * Constructor.
     *
     * @param parameterName The HTML parameter name of the attribute
     * @param label The label text to be used on a form
     * @param isRequireed If the attribute is required or not. Could be indicated on
     *                    the form and enforced with a validation listener
     * @param attributeType The data type of the attribute. String will be used as default in
     *                      other constructors
     */
    public AttributeMetaData(String parameterName,
                             String label,
                             boolean isRequired,
                             boolean isMultiple,
                             AttributeType attributeType) {

        m_parameterName = parameterName;
        m_label = label;
        m_isRequired = isRequired;
        m_isMultiple = isMultiple;
        m_attributeType = attributeType;
    }

    //*** Attribute methods

    public String getParameterName() {
        return m_parameterName;
    }

    public void setLabel(String label) {
        m_label = label;
    }

    public String getLabel() {
        return m_label;
    }

    public void isRequired(boolean isRequired) {
        m_isRequired = isRequired;
    }

    public boolean isRequired() {
        return m_isRequired;
    }

    public void isMultiple(boolean isMultiple) {
        m_isMultiple = isMultiple;
    }

    public boolean isMultiple() {
        return m_isMultiple;
    }

    public void setAttributeType(AttributeType attributeType) {
        m_attributeType = attributeType;
    }

    /**
     * Will return null if no attribute type has been set
     */
    public AttributeType getAttributeType() {
        return m_attributeType;
    }

    /**
     * This method may return null.
     */
    public ParameterModel getParameterModel() {

        ParameterModel parameterModel = null;

        if (m_attributeType != null) {
            Class parameterClass = m_attributeType.getParameterModelClass();

            if (parameterClass != null) {
                try {
                    parameterModel =
                        (ParameterModel)FormBuilderUtil.instantiateObjectOneArg(parameterClass.getName(), m_parameterName);
                } catch (Throwable t) {
                    // Let the parameter model be null
                }
            }
        }

        return parameterModel;
    }
}
