/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.util;

/**
 * <p>A class for Bebop size parameters.</p>
 *
 * <blockquote><pre>
 * private Page buildSomePage() {
 *     Page page = new Page("Some Page");
 *
 *     // Put a 10-pixel margin around the contents of this page.
 *     page.setMargin(new Size(10));
 *
 *     // Or, instead, put a 10% margin around it.
 *     page.setMargin(new Size(10, UNIT_PERCENT));
 *
 *     page.lock();
 *
 *     return page;
 * }
 * </pre></blockquote>
 *
 * @author Justin Ross
 * @author Jim Parsons
 * @author Christian
 * Brechb&uuml;hler
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/util/Size.java#8 $
 */
public class Size {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/util/Size.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    private float m_scalar;
    private String m_unitAbbreviation = "";

    /**
     * Constant for describing sizes in pixels.
     */
    public static final int UNIT_PIXEL = 1;

    /**
     * Constant for describing a component in terms of percent size
     * relative to its container.
     */
    public static final int UNIT_PERCENT = 2;

    /**
     * Construct a new Size.  Classes extending Size should call super
     * with the abbreviation of their unit.
     *
     * @param scalar a simple magnitude.  Note that this value may
     * be negative.
     * @param unitAbbreviation an unit abbreviation for use when the
     * size is printed.
     * @pre unitAbbreviation != null
     */
    protected Size(float scalar, String unitAbbreviation) {
        m_scalar = scalar;
        m_unitAbbreviation = unitAbbreviation;
    }

    /**
     * Construct a new Size in pixels.
     *
     * @param numPixels a simple magnitude.  Note that this value may
     * be negative.
     */
    public Size(int numPixels) {
        this((float)(numPixels), "");
    }

    /**
     * Construct a new Size using the type indicated in unitEnum.
     * unitEnum is any of the UNIT_* constants defined in this class.
     *
     * @param scalar a simple magnitude.  Note that this value may be
     * negative.
     * @param unitEnum a unit type.
     */
    public Size(float scalar, int unitEnum) {
        m_scalar = scalar;

        if (unitEnum == UNIT_PIXEL) {
            m_unitAbbreviation = "";
        } else if (unitEnum == UNIT_PERCENT) {
            m_unitAbbreviation = "%";
        } else {
            throw new IllegalArgumentException
                ("Bad argument for unitEnum in Size constructor.");
        }
    }

    /**
     * Return the size as a string.  This string will be used in
     * writing the style attributes of Bebop XML.
     *
     * @return this Size as a string for inclusion in XML.
     * @post return != null */
    public String toString() {
        String sizeAsString = Float.toString(m_scalar) + m_unitAbbreviation;

        return sizeAsString;
    }
}
