/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop;

import java.util.Collections;
import java.util.Iterator;

import com.arsdigita.bebop.util.Attributes;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.util.Assert;
import com.arsdigita.xml.Element;

/**
 * A simple implementation of the Component interface.
 *
 *
 * @author David Lutterkort
 * @author Stanislav Freidin
 * @author Rory Solomon
 * @author Uday Mathur
 *
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/SimpleComponent.java#15 $
 */
public class SimpleComponent extends Completable
    implements Component, Cloneable {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/SimpleComponent.java#15 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    private boolean m_locked;

    /**
     * The Attribute object is protected to make
     * it easier for the Form Builder service to persist the SimpleComponent.
     * Locking violation is not a problem since if the SimpleComponent is locked
     * then the Attribute object will also be locked.
     */
    protected Attributes m_attr;

    private String m_key = null;        // name mangling key

    /**
     * Clones a component. The clone is not locked and has its own set of
     * attributes.
     * @return the clone of a component.
     * @post ! ((SimpleComponent) return).isLocked()
     */
    public Object clone() throws CloneNotSupportedException {
        SimpleComponent result = (SimpleComponent) super.clone();
        if ( m_attr != null ) {
            result.m_attr = (Attributes) m_attr.clone();
        }
        result.m_locked = false;
        return result;
    }

    public void register(Page p) {
        return;
    }

    public void register(Form f, FormModel m) {
        return;
    }

    /**
     * Does processing that is special to the component
     * receiving the click.
     * @param state the current page state
     */
    public void respond(PageState state)
        throws javax.servlet.ServletException { }

    public Iterator children() {
        return Collections.EMPTY_LIST.iterator();
    }

    /** Adds [J]DOM nodes for this component.  Specifically for
     *  base class SimpleComponent, does nothing.
     */
    public void generateXML(PageState state, Element p) {
        return;
    }

    public final boolean isLocked() {
        return m_locked;
    }

    public void lock () {
        if (m_attr != null) {
            m_attr.lock();
        }
        m_locked = true;
    }

    /**
     * Unlocks this component.  Package visibility is intentional; the
     * only time a component should be unlocked is when it's pooled and
     * gets locked because it's put into a page.  It needs to be unlocked
     * when the instance is recycled.
     */
    void unlock() {
        m_locked = false;

    }

    /* Working with standard component attributes */

    /**
     * Gets the class attribute.
     * @return the class attribute.
     */
    public String getClassAttr() {
        return getAttribute(CLASS);
    }

    /**
     * Sets the class attribute.
     * @param theClass a valid <a
     * href="http://www.w3.org/TR/2000/REC-xml-20001006#NT-Name">XML name</a>
     */
    public void setClassAttr(String theClass) {
        Assert.assertNotLocked(this);
        setAttribute(CLASS, theClass);
    }

    /**
     * Gets the style attribute.
     * @return the style attribute.
     */
    public String getStyleAttr() {
        return getAttribute(STYLE);
    }

    /**
     * Sets the style attribute. <code>style</code> should be a valid CSS
     * style, since its value will be copied verbatim to the output and
     * appear as a <tt>style</tt> attribute in the top level XML or HTML
     * output element.
     *
     * @param style a valid CSS style description for use in the
     *   <tt>style</tt> attribute of an HTML tag
     * @see <a href="#standard">Standard Attributes</a>
     */
    public void setStyleAttr(String style) {
        Assert.assertNotLocked(this);
        setAttribute(STYLE, style);
    }

    /**
     * Gets the <tt>id</tt> attribute.
     * @return the <tt>id</tt> attribute.
     * @see #setIdAttr(String id)
     */
    public String getIdAttr() {
        return getAttribute(ID);
    }

    /**
     * Sets the <tt>id</tt> attribute. <code>id</code>
     * should be an <a
     * href="http://www.w3.org/TR/2000/REC-xml-20001006#NT-Name">XML name</a>
     * that is unique within the {@link Page Page} in which this component is
     * contained. The value of <code>id</code> is copied literally to the
     * output and not used for internal processing.
     *
     * @param id a valid XML identifier
     * @see <a href="#standard">Standard Attributes</a>
     */
    public void setIdAttr(String id) {
        Assert.assertNotLocked(this);
        setAttribute(ID, id);
    }

    /* Methods for attribute management */

    /**
     * Sets an attribute. Overwrites any old values. These values are used to
     * generate attributes for the top level XML or HTML element that is
     * output from this component with {@link #generateXML generateXML}.
     *
     * @pre name != null
     * @post getAttribute(name) == value
     *
     * @param name attribute name, case insensitive
     * @param value new attribute value
     */
    final protected void setAttribute(String name, String value) {
        Assert.assertNotLocked(this);
        if (m_attr == null) {
            m_attr = new Attributes();
        }
        m_attr.setAttribute(name, value);
    }

    /**
     * Gets the value of an attribute.
     *
     * @pre name != null
     *
     * @param name attribute name, case insensitive
     * @return the string value previously set with {@link #setAttribute
     *   setAttribute}, or <code>null</code> if none was set.
     * @see #setAttribute
     */
    final protected String getAttribute(String name) {
        return (m_attr == null) ? null : m_attr.getAttribute(name);
    }

    /**
     * Adds the attributes set with {@link #setAttribute setAttribute} to the
     * element <code>target</code>. The attributes set with
     * <code>exportAttributes</code> overwrite attributes with identical names
     * that <code>target</code> might already have.
     *
     * @pre target != null
     *
     * @param target element to which attributes are added
     * @see #setAttribute
     */
    final protected void exportAttributes(com.arsdigita.xml.Element target) {
        if (m_attr != null) {
            m_attr.exportAttributes(target);
        }
        if (Kernel.getConfig().isDebugEnabled()) {
            target.addAttribute("bebop:classname", getClass().getName(),
                                BEBOP_XML_NS);
        }
    }

    /**
     * Returns <code>true</code> if any attributes have been set.
     * @return <code>true</code> if any attributes have been set;
     * <code>false</code> otherwise.
     */
    final protected boolean hasAttributes() {
        return m_attr != null;
    }

    /*
     * Set an arbitrary meta data attribute on the component.
     * The name of the attribute in the XML will be prefixed
     * with the string 'metadata.'
     */
    final public void setMetaDataAttribute(String name, String value) {
        setAttribute("metadata." + name, value);
    }

    final public String getMetaDataAttribute(String name) {
        return getAttribute("metadata." + name);
    }

    /**
     * Supplies a key for parameter name mangling.
     * @param key the key to mangle
     */
    public Component setKey(String key) {
        Assert.assertNotLocked(this);
        if (key.charAt(0) >= 0 && key.charAt(0) <= 9) //TODO: be more strict.
            throw new IllegalArgumentException
                ("key \"" + key + "\" must not start with a digit.");
        m_key = key;
        return this;
    }

    /**
     * Retrieves a key for parameter name mangling.
     * @return a key for parameter name mangling.
     */
    public final String getKey() {
        return m_key;
    }

    public boolean isVisible(PageState s) {
        return s.isVisible(this);
    }

    public void setVisible(PageState s, boolean  v) {
        s.setVisible(this, v);
    }

}
