/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop;

import com.arsdigita.bebop.table.TableCellRenderer;
import com.arsdigita.bebop.table.TableModel;
import com.arsdigita.bebop.table.TableModelBuilder;
import com.arsdigita.bebop.util.GlobalizationUtil;
import com.arsdigita.globalization.GlobalizedMessage;
import com.arsdigita.util.LockableImpl;


/**
 * Displays a list of label-value pairs that
 * represent the properties of some object. For example, a
 * typical <code>PropertySheet</code> may look like this:
 * <p>
 * <table cellpadding=4 cellspacing=0 border=0>
 * <tr><th>First Name:</th><td>Stanislav</td></tr>
 * <tr><th>Last Name:</th><td>Freidin</td></tr>
 * <tr><th>Mission:</th><td>Sleep</td></tr>
 * </table>
 * <p>
 * This class relies on the {@link PropertySheetModelBuilder} to
 * supply it with the right {@link PropertySheetModel} during
 * each request. It is up to the user to provide the right
 * builder.
 * <p>
 *
 * @author Stanislav Freidin
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/PropertySheet.java#11 $
 *
 */
public class PropertySheet extends Table {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/PropertySheet.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private PropertySheetModelBuilder m_builder;

    /**
     * Constructs a new <code>PropertySheet</code>.
     *
     * @param modelBuilder the property sheet model builder
     *   that is responsible for building the property
     *   sheet model
     *
     */
    public PropertySheet(PropertySheetModelBuilder modelBuilder) {
        super(new PSTMBAdapter(modelBuilder), new Object[]{"Label", "Value"});
        super.setHeader(null);

        super.getColumn(0).setCellRenderer(new GlobalizedLabelCellRenderer(Label.BOLD));
        super.getColumn(1).setCellRenderer(new StringLabelCellRenderer(null));
    }

    /**
     * Constructs a new <code>PropertySheet</code> and sets the output
     * escape value.
     *
     * @param modelBuilder the property sheet model builder
     *   that is responsible for building the property
     *   sheet model
     * @param valueOutputEscape the value of the label-value
     *   pair's output-escaping
     *
     */
    public PropertySheet(PropertySheetModelBuilder modelBuilder, boolean valueOutputEscape) {
        super(new PSTMBAdapter(modelBuilder), new Object[]{"Label", "Value"});
        super.setHeader(null);

        super.getColumn(0).setCellRenderer(new GlobalizedLabelCellRenderer(Label.BOLD));
        super.getColumn(1).setCellRenderer(new StringLabelCellRenderer(valueOutputEscape));
    }

    /**
     * Returns the {@link PropertySheetModelBuilder}.
     * @return the {@link PropertySheetModelBuilder}.
     */
    public PropertySheetModelBuilder getPropertySheetModelBuilder() {
        return m_builder;
    }

    // Convert a PropertySheetModelBuilder to a TableModelBuilder
    private static class PSTMBAdapter
        extends LockableImpl implements TableModelBuilder {

        private PropertySheetModelBuilder m_builder;

        public PSTMBAdapter(PropertySheetModelBuilder b) {
            m_builder = b;
        }

        public TableModel makeModel(Table t, PageState s) {
            return new TableModelAdapter(
                                         m_builder.makeModel((PropertySheet)t, s)
                                         );
        }

        public void lock() {
            m_builder.lock();
            super.lock();
        }
    }

    // Wraps a PropertySheetModel
    private static class TableModelAdapter implements TableModel {

        private PropertySheetModel m_model;
        private int m_row;

        public TableModelAdapter(PropertySheetModel model) {
            m_model = model;
            m_row = -1;
        }

        public int getColumnCount() { return 2; }

        public boolean nextRow() {
            m_row++;
            return m_model.nextRow();
        }

        public Object getElementAt(int columnIndex) {
            if(columnIndex == 0) {
                return m_model.getGlobalizedLabel();
            } else {
                return m_model.getValue();
            }
        }

        public Object getKeyAt(int columnIndex) {
            return new Integer(m_row);
        }

        public PropertySheetModel getPSModel() {
            return m_model;
        }
    };

    // Renders strings as labels
    public static class StringLabelCellRenderer implements TableCellRenderer {

        private String m_weight;
        private boolean m_outputEscape = false;

        public StringLabelCellRenderer(String weight) {
            m_weight = weight;
        }

        public StringLabelCellRenderer(boolean outputEscape) {
            m_outputEscape = outputEscape;
        }

        public Component getComponent(Table table, PageState state, Object value,
                                      boolean isSelected, Object key,
                                      int row, int column) {
            Label t = getLabel(value);
            //t.setOutputEscaping(false);
            t.setOutputEscaping(m_outputEscape);
            if(m_weight != null) {
                t.setFontWeight(m_weight);
            }
            return t;
        }

        protected Label getLabel(Object value) {
            return new Label((String)value);
        }
    }

    // Renders strings as labels
    public static class GlobalizedLabelCellRenderer extends StringLabelCellRenderer {

        public GlobalizedLabelCellRenderer(String weight) {
            super(weight);
        }

        public GlobalizedLabelCellRenderer(boolean outputEscape) {
            super(outputEscape);
        }

        protected Label getLabel(Object value) {
            return new Label((GlobalizedMessage)value);
        }
    }

    /**
     * An empty {@link PropertySheetModel}.
     */
    public static final PropertySheetModel EMPTY_MODEL =
        new PropertySheetModel() {
            public boolean nextRow() { return false; }
            public String getLabel() {
                throw new IllegalStateException("The model is empty");
            }
            public GlobalizedMessage getGlobalizedLabel() {
                throw new IllegalStateException((String) GlobalizationUtil.globalize("bebop.the_model_is_empty").localize());
            }
            public String getValue() {
                throw new IllegalStateException("The model is empty");
            }
        };

}
