/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel;

import com.arsdigita.persistence.*;

import com.arsdigita.db.Sequences;

import com.arsdigita.domain.DataObjectNotFoundException;

import com.arsdigita.tools.junit.framework.BaseTestCase;
import junit.framework.Test;
import junit.framework.TestSuite;

import java.math.BigDecimal;
import java.sql.*;

// Logging Support.

/**
 * Tests basic functionality of the UserAuthentication class
 *
 *
 * @author Phong Nguyen
 * @version 1.0
 * @see com.arsdigita.kernel.UserAuthentication
 */
public class UserAuthenticationTest extends BaseTestCase {

    public final static String versionId = "$Id: //core-platform/dev/test/src/com/arsdigita/kernel/UserAuthenticationTest.java#9 $";

    private Session m_ssn;

    /**
     * Constructs a UserAuthenticationTest with the specified name.
     *
     * @param name Test case name.
     **/
    public UserAuthenticationTest( String name ) {
        super( name );
    }

    public void setUp()  {
        try {
            m_ssn = SessionManager.getSession();
        } catch (Exception e) {
            e.printStackTrace(System.err);
            fail(e.getMessage());
        }
    }

    public void tearDown() {
    }


    /**
     * Tests database retrieval
     **/
    public void testUserAuthentication() throws Exception {

        User user = new User();
        BigDecimal idval = user.getID();
        String screenName = "phongn76_" + idval.toString();
        String givenName = "Phong";
        String familyName = "Nguyen";
        String emailAddress = "phong" + idval + "@arsdigita.com";
        String password = "acsrules";
        String passwordQuestion = "What is your lucky number?";
        String passwordAnswer = "7";

        // create a new user
        user.setScreenName(screenName);
        user.getPersonName().setGivenName(givenName);
        user.getPersonName().setFamilyName(familyName);
        user.setPrimaryEmail(new EmailAddress(emailAddress));
        user.save();

        // retrieve the user
        OID oid = new OID(User.BASE_DATA_OBJECT_TYPE, idval);
        user = new User(oid);
        assertEquals("The given name for the user is not correct.",
                     givenName, user.getPersonName().getGivenName());
        assertEquals("The family name for the user is not correct.",
                     familyName, user.getPersonName().getFamilyName());
        assertEquals("The screen name for the user is not correct.",
                     screenName, user.getScreenName());
        assertEquals("The primary email for the user is not correct.",
                     emailAddress, user.getPrimaryEmail().getEmailAddress());

        // set authentication data for the user
        UserAuthentication userAuth = UserAuthentication.createForUser(user);
        userAuth.setPassword(password);
        userAuth.setPasswordQuestion(passwordQuestion);
        userAuth.setPasswordAnswer(passwordAnswer);
        user.getPersonName().setGivenName(givenName);
        user.getPersonName().setFamilyName(familyName);
        userAuth.save();

        // retrieving authentication data
        userAuth = UserAuthentication.retrieveForUser(oid);
        assertEquals("The password question is not correct.",
                     passwordQuestion, userAuth.getPasswordQuestion());
        assertTrue("The password \"" + password + "\" should have been the valid " +
                   "password.", userAuth.isValidPassword(password));
        assertTrue("The answer \"" + passwordAnswer + "\" should have been the " +
                   "valid password.", userAuth.isValidAnswer(passwordAnswer));

        // retrieving authentication and user by login name
        userAuth = UserAuthentication.retrieveForLoginName(emailAddress);
        user = userAuth.getUser();
        assertEquals("The given name for the user is not correct.",
                     givenName, user.getPersonName().getGivenName());
        assertEquals("The family name for the user is not correct.",
                     familyName, user.getPersonName().getFamilyName());
        assertEquals("The screen name for the user is not correct.",
                     screenName, user.getScreenName());
        assertEquals("The primary email for the user is not correct.",
                     emailAddress, user.getPrimaryEmail().getEmailAddress());
        assertEquals("The password question is not correct.",
                     passwordQuestion, userAuth.getPasswordQuestion());
        assertTrue("The password \"" + password + "\" should have been the valid " +
                   "password.", userAuth.isValidPassword(password));
        assertTrue("The answer \"" + passwordAnswer + "\" should have been the " +
                   "valid password.", userAuth.isValidAnswer(passwordAnswer));

        // Make sure exception thrown with invalid login name
        try {
            UserAuthentication.retrieveForLoginName("This is a bogus login name");
            fail("No exception thrown when retrieving a user " +
                 "authentication record for a bogus email address");
        } catch (DataObjectNotFoundException e) {
        }

    }

    public static Test suite() {
        //
        // Reflection is used here to add all
        // the testXXX() methods to the suite.
        //
        return new TestSuite(UserAuthenticationTest.class);
    }

    public static void main(String args[]) {
        junit.textui.TestRunner.run(suite());
    }
}
