/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.globalization;

import com.arsdigita.tools.junit.framework.BaseTestCase;
import com.arsdigita.dispatcher.TestUtils;
import com.arsdigita.util.HttpServletDummyRequest;

import java.util.Locale;
import java.util.ResourceBundle;

public class LocaleNegotiatorTest extends BaseTestCase {

    public static final String versionId = "$Id: //core-platform/dev/test/src/com/arsdigita/globalization/LocaleNegotiatorTest.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    public LocaleNegotiatorTest(String name) {
        super(name);
    }

    public static final String DefaultCharsetName = "ISO-8859-1";

    public void FAILStest0() throws Exception {
        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle0",
                                                   null, null, null);

        ResourceBundle rb = ResourceBundle.getBundle(
                                                     "com.arsdigita.globalization.TestBundle0");

        assertEquals("locale", Locale.getDefault(), ln.getLocale());
        assertEquals("charset", DefaultCharsetName, ln.getCharset());
        assertEquals("bundle", rb, ln.getBundle());
    }

    public void FAILStest1() throws Exception {
        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle0",
                                                   "", null, null);

        ResourceBundle rb = ResourceBundle.getBundle(
                                                     "com.arsdigita.globalization.TestBundle0");

        assertEquals("locale", Locale.getDefault(), ln.getLocale());
        assertEquals("charset", DefaultCharsetName, ln.getCharset());
        assertEquals("bundle", rb, ln.getBundle());
    }

    public void FAILStest2() throws Exception {
        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle0",
                                                   null, "", null);

        ResourceBundle rb = ResourceBundle.getBundle(
                                                     "com.arsdigita.globalization.TestBundle0");

        assertEquals("locale", Locale.getDefault(), ln.getLocale());
        assertEquals("charset", DefaultCharsetName, ln.getCharset());
        assertEquals("bundle", rb, ln.getBundle());
    }

    private Charset makeCharset(String name) {
        Charset cs = new Charset();
        cs.setCharset(name);
        cs.save();
        return cs;
    }

    private Locale makeLocale(
                              String language, String country, String variant,
                              Charset cs)
    {
        com.arsdigita.globalization.Locale l;

        l = new com.arsdigita.globalization.Locale(
                                                   language, country, variant);
        l.setDefaultCharset(cs);
        l.save();

        return l.toJavaLocale();
    }

    private void verify(LocaleNegotiator ln, Locale l,
                        String charsetName, String bundleName, Locale bundleLocale)
    {
        assertEquals("locale", l, ln.getLocale());
        assertEquals("charset", charsetName, ln.getCharset());
        assertEquals("bundle",
                     ResourceBundle.getBundle(
                                              "com.arsdigita.globalization.TestBundle" + bundleName,
                                              bundleLocale),
                     ln.getBundle());
    }

    public void FAILStest3() throws Exception {
        Charset cs0 = makeCharset("cs0");
        Locale l =
            makeLocale("l0", "",   "",   cs0);
        makeLocale("l0", "C0", "",   cs0);

        ClientLocaleProvider clp = (ClientLocaleProvider)
            LocaleNegotiator.getClientLocaleProvider();
        if (clp != null) {
            clp.setTargetBundle("com.arsdigita.globalization.TestBundle3");
            clp.setAcceptLanguages("l0");
        }

        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle3", "", "", null
                                                   );

        verify(ln, l, "cs0", "3", l);
    }

    public void FAILStest4() throws Exception {
        Charset cs0 = makeCharset("cs0");
        makeLocale("l0", "",   "",   cs0);
        Locale l =
            makeLocale("l0", "C0", "",   cs0);

        ClientLocaleProvider clp = (ClientLocaleProvider)
            LocaleNegotiator.getClientLocaleProvider();
        if (clp != null) {
            clp.setTargetBundle("com.arsdigita.globalization.TestBundle3");
            clp.setAcceptLanguages("l0-C0");
        }

        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle3", "", "", null
                                                   );

        verify(ln, l, "cs0", "3", l);
    }

    public void FAILStest5() throws Exception {
        Charset cs0 = makeCharset("cs0");

        Locale l = Locale.getDefault();

        makeLocale("l0", "",   "",   cs0);
        makeLocale("l0", "C0", "",   cs0);

        ClientLocaleProvider clp = (ClientLocaleProvider)
            LocaleNegotiator.getClientLocaleProvider();
        if (clp != null) {
            clp.setTargetBundle("com.arsdigita.globalization.TestBundle3");
            clp.setAcceptLanguages("l0; q=0, *");
        }

        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle3", "", "", null
                                                   );

        verify(ln, l, "ISO-8859-1", "3", Locale.getDefault());
    }

    public void FAILStest6() throws Exception {
        Charset cs0 = makeCharset("cs0");
        Locale l =
            makeLocale("l0", "",   "",   cs0);
        makeLocale("l0", "C0", "",   cs0);

        ClientLocaleProvider clp = (ClientLocaleProvider)
            LocaleNegotiator.getClientLocaleProvider();
        if (clp != null) {
            clp.setTargetBundle("com.arsdigita.globalization.TestBundle3");
            clp.setAcceptLanguages("l1, l0");
        }

        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle3", "", "", null
                                                   );

        verify(ln, l, "cs0", "3", l);
    }

    public void test7() throws Exception {
        Charset cs0 = makeCharset("cs0");
        makeLocale("l0", "",   "",   cs0);
        final Locale l =
            makeLocale("l0", "C0", "",   cs0);

        ClientLocaleProvider clp = (ClientLocaleProvider)
            LocaleNegotiator.getClientLocaleProvider();
        if (clp != null) {
            clp.setTargetBundle("com.arsdigita.globalization.TestBundle3");
            clp.setAcceptLanguages("en");
        }

        LocaleNegotiator.setApplicationLocaleProvider(
                                                      new LocaleProvider() {
                                                          public Locale getLocale() {
                                                              return l;
                                                          }
                                                      }
                                                      );

        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle3", "", "", null
                                                   );

        verify(ln, l, "cs0", "3", l);
    }

    public void test8() throws Exception {
        Charset cs0 = makeCharset("cs0");
        final Locale l =
            makeLocale("l0", "",   "",   cs0);
        makeLocale("l0", "C0", "",   cs0);

        LocaleNegotiator.setApplicationLocaleProvider(
                                                      new LocaleProvider() {
                                                          public Locale getLocale() {
                                                              return l;
                                                          }
                                                      }
                                                      );

        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle8", "", "", null
                                                   );

        // Should this expect TestBundle8 instead of TestBundle8_en?
        verify(ln, l, "cs0", "8", new Locale("en", ""));
    }

    public void test9() throws Exception {
        TestUtils.setRequest(new HttpServletDummyRequest());
        Charset cs0 = makeCharset("cs0");
        Locale l =
            makeLocale("l0", "",   "",   cs0);
        makeLocale("l0", "C0", "",   cs0);

        ClientLocaleProvider clp = (ClientLocaleProvider)
            LocaleNegotiator.getClientLocaleProvider();
        if (clp != null) {
            clp.setTargetBundle("com.arsdigita.globalization.TestBundle9");
            clp.setAcceptLanguages("en; q=0.9, l0; q=1.0");
        }

        LocaleNegotiator ln = new LocaleNegotiator(
                                                   "com.arsdigita.globalization.TestBundle9", "", "", null
                                                   );

        verify(ln, l, "cs0", "9", l);
    }
}
