/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.web;

import com.arsdigita.kernel.ACSObject;
import com.arsdigita.kernel.ACSObjectCollection;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.util.Assert;
import org.apache.log4j.Logger;

/**
 * <p>A collection of Applications.  This class includes methods to
 * filter and order the results of retrieving applications.</p>
 *
 * @see com.arsdigita.web.Application
 * @see com.arsdigita.web.ApplicationType
 * @see com.arsdigita.web.ApplicationTypeCollection
 * @author Justin Ross &lt;<a href="mailto:jross@redhat.com">jross@redhat.com</a>&gt;
 * @version $Id: //core-platform/dev/src/com/arsdigita/web/ApplicationCollection.java#7 $
 */
public class ApplicationCollection extends ACSObjectCollection {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/web/ApplicationCollection.java#7 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (ApplicationCollection.class);

    public ApplicationCollection(DataCollection dataCollection) {
        super(dataCollection);
    }

    public void filterToApplicationType(String type) {
        m_dataCollection.addEqualsFilter(ACSObject.OBJECT_TYPE, type);
    }

    public void filterToWorkspaceApplications() {
        m_dataCollection.addEqualsFilter
            ("resourceType.isWorkspaceApplication", Boolean.TRUE);
    }

    public void filterToPortalSiteApplications() {
        m_dataCollection.addEqualsFilter
            ("resourceType.isWorkspaceApplication", Boolean.TRUE);
    }

    public void filterToHasFullPageView() {
        m_dataCollection.addEqualsFilter
            ("resourceType.hasFullPageView", Boolean.TRUE);
    }

    public void filterToHasEmbeddedView() {
        m_dataCollection.addEqualsFilter
            ("resourceType.hasEmbeddedView", Boolean.TRUE);
    }

    /**
     * Orders by the title of the application type.
     **/
    public void orderByTypeTitle() {
        m_dataCollection.addOrder("resourceType.title");
    }

    public void orderByTitle() {
        m_dataCollection.addOrder("title");
    }

    public void orderByParentTitle() {
        m_dataCollection.addOrder("parentResource.title");
    }

    /**
     * Get the current item as an Application domain object.
     *
     * @return an Application domain object.
     * @post return != null
     */
    public Application getApplication() {
        DataObject dataObject = m_dataCollection.getDataObject();

        Application application =
            Application.retrieveApplication(dataObject);

        Assert.assertNotNull(application, "application");

        return application;
    }

    /**
     * Get the title for the application for the current row.
     *
     * @return the title of this application.
     * @post return != null
     */
    public String getTitle() {
        String title = (String)m_dataCollection.get("title");

        Assert.assertNotNull(title, "title");

        return title;
    }

    /**
     * Returns the title of the parent application.
     *
     * @return The title of the parent application.
     **/
    public String getParentTitle() {
        return (String) m_dataCollection.get("parentResource.title");
    }

    public String getDescription() {
        return (String) m_dataCollection.get("description");
    }

    /**
     * Get the primary URL for the current row's application.
     *
     * @return the primary URL of this application.
     * @post return != null
     */
    public String getPrimaryURL() {
        String primaryURL = (String)m_dataCollection.get("primaryURL");

        Assert.assertNotNull(primaryURL, "primaryURL");

        return primaryURL;
    }
}
