/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.ui.login;

import com.arsdigita.bebop.PageState;
import com.arsdigita.dispatcher.DispatcherHelper;
import com.arsdigita.globalization.GlobalizedMessage;
import com.arsdigita.web.ReturnSignal;
import java.io.IOException;
import org.apache.log4j.Logger;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Provides helper functions for the login UI.
 *
 * @author Sameer Ajmani
 **/
public class LoginHelper {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/ui/login/LoginHelper.java#13 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger(LoginHelper.class);

    public static final String RETURN_URL_PARAM_NAME = "return_url";

    /**
     * Returns the name of the login UI resource bundle
     *
     * @return the name of the login UI resource bundle
     **/
    static String getBundleBaseName() {
        return "com.arsdigita.ui.login.SubsiteResources";
    }

    /**
     * Returns a new GlobalizedMessage constructed with the given
     * parameters and the login UI resource bundle.
     *
     * @return a new GlobalizedMessage
     **/
    static GlobalizedMessage getMessage(String key, Object[] args) {
        return new GlobalizedMessage(key, getBundleBaseName(), args);
    }

    /**
     * Returns a new GlobalizedMessage constructed with the given
     * parameters and the login UI resource bundle.
     *
     * @return a new GlobalizedMessage
     **/
    static GlobalizedMessage getMessage(String key) {
        return new GlobalizedMessage(key, getBundleBaseName());
    }

    /**
     * Constructs a new GlobalizedMessage with the given parameters and the
     * login UI resource bundle, then localizes the message with the given
     * request.
     *
     * @return the localized String
     **/
    static String localize(String key, Object[] args,
                           HttpServletRequest request) {
        return (String)getMessage(key, args).localize(request);
    }

    /**
     * Constructs a new GlobalizedMessage with the given parameters and the
     * login UI resource bundle, then localizes the message with the given
     * request.
     *
     * @return the localized String
     **/
    static String localize(String key, HttpServletRequest request) {
        return (String)getMessage(key).localize(request);
    }

    /**
     * Redirect the client to the URL stored in the return_url request
     * parameter, or, if that parameter is not set, redirect to the given
     * default URL.
     *
     * @param state the current page state
     * @param def the default URL
     **/
    public static void doReturn(PageState state, String def) {
        throw new ReturnSignal(state.getRequest(), def);
    }

    /**
     * Redirect the client to the given URL unless the response has
     * already been committed. Wrapper for {@link
     * #sendRedirect(HttpServletRequest, HttpServletResponse, String)}
     * that pulls out the request and response from the PageState.
     *
     * @throws IOException if the redirect fails.
     **/
    public static void sendRedirect(PageState state, String url)
        throws IOException {
        sendRedirect(state.getRequest(), state.getResponse(), url);
    }

    /**
     * Redirect the client to the given URL unless the response has already
     * been committed.  Aborts further request processing.
     *
     * @throws IOException if the redirect fails.
     **/
    public static void sendRedirect(HttpServletRequest request,
                                    HttpServletResponse response,
                                    String url)
        throws IOException {

        if (!response.isCommitted()) {
            s_log.debug("Redirecting to: "+url);
            DispatcherHelper.sendRedirect(request, response, url);
            response.flushBuffer();
            DispatcherHelper.abortRequest();
        } else {
            s_log.debug("Redirect failed because "
                        +"response already committed");
        }
    }
}
