/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.toolbox.ui;

import com.arsdigita.bebop.event.RequestEvent;
import com.arsdigita.bebop.event.RequestListener;

import com.arsdigita.dispatcher.AccessDeniedException;


import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.Resource;
import com.arsdigita.kernel.Party;
import com.arsdigita.kernel.permissions.PermissionDescriptor;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.web.LoginSignal;

import com.arsdigita.ui.login.UserAuthenticationListener;

import java.io.IOException;
import org.apache.log4j.Logger;

/**
 * <p>
 * A request listener that checks a privilege against
 * the current resource as accessed from Kernel.getContext().getResource()
 * For example, to restrict a page to admin users only:
 * </p>
 * <pre>
 *  Page page = PageFactory.buildPage("myapp", "My Admin Page");
 *  page.addRequestListener(new ApplicationAuthenticationListener());
 *  page.lock();
 * </pre>
 */
public class ApplicationAuthenticationListener implements RequestListener {

    private static final Logger s_log = Logger.getLogger
        (ApplicationAuthenticationListener.class);

    private PrivilegeDescriptor m_privilege;

    /**
     * Creates a listener that checks for ADMIN privilege
     */
    public ApplicationAuthenticationListener() {
        this(PrivilegeDescriptor.ADMIN);
    }
    
    /**
     * Creates a listener that checks for an arbitrary
     * privilege
     * @param privilege the privilege to check for
     */
    public ApplicationAuthenticationListener(PrivilegeDescriptor privilege) {
        m_privilege = privilege;
    }

    /**
     * Checks whether the user is logged in.  If not, redirects the client
     * to the login page.
     */
    public void pageRequested(RequestEvent event) {
        Party party = Kernel.getContext().getParty();

        Resource resource = Kernel.getContext().getResource();

        PermissionDescriptor permDescriptor =
            new PermissionDescriptor(m_privilege,
                                     resource,
                                     party);
        
        if (!PermissionService.checkPermission(permDescriptor)) {
            if (party == null) {
                throw new LoginSignal(event.getPageState().getRequest());
            }
            throw new AccessDeniedException(
                "user " + party.getOID() + " doesn't have the " + 
                m_privilege.getName() + " privilege on " + 
                resource.getOID());
        }
    }
}
