/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search.intermedia;


import com.arsdigita.search.QueryEngine;
import com.arsdigita.search.Search;
import com.arsdigita.search.IndexerType;
import com.arsdigita.search.BaseDocument;
import com.arsdigita.search.ResultSet;
import com.arsdigita.search.QuerySpecification;
import com.arsdigita.search.FilterSpecification;
import com.arsdigita.search.FilterType;

import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.DataObject;

import com.arsdigita.util.Assert;

import java.util.Date;
import java.util.Iterator;
import java.util.Locale;
import java.util.Collections;

import java.math.BigDecimal;

import org.apache.log4j.Logger;


class DataQueryResultSet implements ResultSet {

    private static final Logger s_log = 
        Logger.getLogger(DataQueryResultSet.class);
    
    private DataQuery m_query;
    private long m_count;
    private long m_queryTime;

    public DataQueryResultSet(DataQuery query) {
        m_query = query;
        Date start = new Date();
        m_count = query.size();
        Date end = new Date();
        
        m_queryTime = end.getTime() - start.getTime();
        
        if (s_log.isDebugEnabled()) {
            s_log.debug("Result set count is " + m_count + 
                        ", query duration " + m_queryTime);
        }
    }

    public Iterator getDocuments(long offset,
                                 long count) {
        if (s_log.isDebugEnabled()) {
            s_log.debug("Paginating at offset " + offset + 
                        " for " + count + " rows");
        }
        if (count == 0) {
            return Collections.EMPTY_LIST.iterator();
        }

        m_query.setRange(new Integer((int)offset+1), 
                         new Integer((int)(offset+count+1)));
        
        return new DataQueryIterator(m_query);
    }
    
    public String getEngine() {
        return IndexerType.INTERMEDIA.getKey();
    }

    public long getCount() {
        return m_count;
    }
    
    public long getQueryTime() {
        return m_queryTime;
    }
    
    public void close() {
        m_query.close();
    }
    
    private class DataQueryIterator implements Iterator {
        
        private DataQuery m_query;
        private BaseDocument m_current;
        private boolean m_hasNext;
        
        public DataQueryIterator(DataQuery query) {
            m_query = query;
            peekNext();
        }
        
        public boolean hasNext() {
            return m_hasNext;
        }
        
        public Object next() {
            Assert.truth(m_hasNext, "hasNext");

            BaseDocument result = m_current;
            peekNext();
            return result;
        }
        
        private void peekNext() {
            m_hasNext = m_query.next();
            if (m_hasNext) {
                m_current = new BaseDocument(
                    new OID((String)m_query.get(BaseQueryEngine.OBJECT_TYPE),
                            (BigDecimal)m_query.get(BaseQueryEngine.OBJECT_ID)),
                    new Locale((String)m_query.get(BaseQueryEngine.LANGUAGE),
                               "us"),
                    (String)m_query.get(BaseQueryEngine.LINK_TEXT),
                    (String)m_query.get(BaseQueryEngine.SUMMARY),
                    null,
                    null,
                    null,
                    null,
                    (BigDecimal)m_query.get(BaseQueryEngine.SCORE)
                );
            } else {
                m_current = null;
            }
        }
        
        public void remove() {
            throw new UnsupportedOperationException("cannot remove items");
        }
    }    
}
