/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search.filters;

import com.arsdigita.search.FilterSpecification;
import com.arsdigita.persistence.metadata.ObjectType;
import com.arsdigita.persistence.metadata.MetadataRoot;

import com.arsdigita.util.Assert;

/**
 * A filter spec for supplying a list of object types
 * to the object type filter type. There are two flags
 * can alter the semantics of the filter. The first 
 * specifies whether the list of an inclusion (white)
 * or exclusion (black) list.
 */
public class ObjectTypeFilterSpecification extends FilterSpecification {

    public final static String TYPES = "types";
    public final static String EXCLUSION = "exclude";
    
    /**
     * Creates a new filter restricting results to a single
     * object type.
     * @param typeName the object type name
     */
    public ObjectTypeFilterSpecification(String typeName) {
        this(lookupTypes(new String[] {typeName}));
    }

    /**
     * Creates a new filter restricting results to a set
     * object types.
     * @param typeName the object type names
     */
    public ObjectTypeFilterSpecification(String[] typeNames) {
        this(lookupTypes(typeNames));
    }

    /**
     * Creates a new filter restricting results to a single
     * object type.
     * @param type the object type
     */
    public ObjectTypeFilterSpecification(ObjectType type) {
        super(new Object[] { TYPES, new ObjectType[] { type } },
              new ObjectTypeFilterType());
    }

    /**
     * Creates a new filter restricting results to a set
     * object type.
     * @param type the object types
     */
    public ObjectTypeFilterSpecification(ObjectType[] types) {
        super(new Object[] { TYPES, types },
              new ObjectTypeFilterType());
    }
    
    /**
     * Returns the set of object types to filter on
     * @return the object type
     */
    public ObjectType[] getTypes() {
        return (ObjectType[])get(TYPES);
    }
    
    /**
     * Sets the flag indicating that the type list is
     * an exclusion list rather than an inclusion list.
     * Default is an inclusion list.
     *
     * @param exclude true to mark as an exclusion list
     */
    public void setExclusion(boolean exclude) {
        set(EXCLUSION, new Boolean(exclude));
    }

    /**
     * Gets the flag indicating that the type list is
     * an exclusion list rather than an inclusion list.
     *
     * @return true if marked as an exclusion list
     */
    public boolean isExclusion() {
        return Boolean.TRUE.equals(get(EXCLUSION));
    }
    
    private static ObjectType[] lookupTypes(String[] typeNames) {
        Assert.exists(typeNames, String[].class);
        ObjectType[] types = new ObjectType[typeNames.length];
        for (int i = 0 ; i < typeNames.length ; i++) {
            types[i] = MetadataRoot.getMetadataRoot().getObjectType(typeNames[i]);
        }
        return types;
    }
}
