/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search;

import com.arsdigita.util.Assert;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;

import org.apache.log4j.Logger;

/**
 * Provides a registry of query engine implementations for 
 * various sets of filters.  Application programmers do not need
 * to access instances of this class directly, rather they
 * should use the <code>process</code> method in the <code>Search</code>
 * class.
 * 
 * @see com.arsdigita.search.QueryEngine
 */
public class QueryEngineRegistry {
    
    private static Map s_engines = new HashMap();

    private static final Logger s_log = 
        Logger.getLogger(QueryEngineRegistry.class);

    static {
        registerEngine(IndexerType.NOOP,
                       new FilterType[] {},
                       new NoopQueryEngine());
    }

    /**
     * Registers a new query engine implementation for an
     * indexer capable of accepting a specific set of filter 
     * types.
     * @param indexer the search engine type
     * @param filters the filter types supported
     * @param engine the engine implementation 
     */
    public static void registerEngine(IndexerType indexer,
                                      FilterType[] filters,
                                      QueryEngine engine) {
        Assert.exists(indexer, IndexerType.class);

        registerEngine(indexer.getKey(), filters, engine);
    }

    /**
     * Registers a new query engine implementation for an
     * indexer capable of accepting a specific set of filter 
     * types.
     * @param indexer the search engine type
     * @param filters the filter types supported
     * @param engine the engine implementation 
     */
    public static void registerEngine(String indexer,
                                      FilterType[] filters,
                                      QueryEngine engine) {
        Assert.exists(indexer, String.class);
        Assert.exists(filters, FilterType.class);
        Assert.exists(engine, QueryEngine.class);
        
        if (s_log.isDebugEnabled()) {
            s_log.debug("Register engine " + engine.getClass() + " for " + indexer);
            for (int i = 0 ; i < filters.length ; i++) {
                s_log.debug("  Filter: " + filters[i].getKey() + " (" + 
                            filters[i].getClass() + ")");
            }
        }
        
        s_engines.put(new EngineKey(indexer, filters), 
                      engine);
    }

    /**
     * Gets the search engine implementation capable of processing
     * the specified set of filters on a given search indexer.
     * @param indexer the search engine type
     * @param filters the filter types requested
     * @return a search engine implementation, or null
     */
    public static QueryEngine getEngine(String indexer, 
                                        FilterType[] filters) {
        Assert.exists(indexer, String.class);
        Assert.exists(filters, FilterType.class);

        if (s_log.isDebugEnabled()) {
            s_log.debug("Lookup engine for " + indexer);
            for (int i = 0 ; i < filters.length ; i++) {
                s_log.debug("  Filter: " + filters[i].getKey() + " (" + 
                            filters[i].getClass() + ")");
            }
        }

        return (QueryEngine)s_engines.get(new EngineKey(indexer, filters));
    }

    private static class EngineKey {
        private String m_indexer;
        private Set m_filters;
        
        public EngineKey(String indexer,
                         FilterType[] filters) {
            m_indexer = indexer;
            m_filters = new HashSet();
            for (int i = 0 ; i < filters.length ; i++) {
                m_filters.add(filters[i]);
            }
        }
        
        public int hashCode() {
            return m_filters.hashCode();
        }

        public boolean equals(Object o) {
            if (!(o instanceof EngineKey)) {
                return false;
            }
            
            EngineKey key = (EngineKey)o;

            if (!m_indexer.equals(key.m_indexer)) {
                return false;
            }
            
            return m_filters.equals(key.m_filters);
        }
    }
}
