/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.runtime;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;

import org.apache.log4j.Logger;

/**
 * The CompoundInitializer class provides a convenient way to group
 * together a number of individual initializers into a single object
 * that implements the Initializer interface. Using this class to
 * group initializers will guarantee that the various init methods on
 * the contained initializers will be invoked in the proper order,
 * i.e. all data init methods will be invoked first in order, followed
 * by all domain init methods, followed by all legacy init methods.
 *
 * @author Rafael H. Schloming &lt;rhs@mit.edu&gt;
 * @version $Revision: #5 $ $Date: 2004/04/07 $
 **/

public class CompoundInitializer implements Initializer {

    private static final Logger s_log = 
        Logger.getLogger(CompoundInitializer.class);

    public final static String versionId = "$Id: //core-platform/dev/src/com/arsdigita/runtime/CompoundInitializer.java#5 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private Logger m_log;

    private final List m_inits = new ArrayList();

    /**
     * Creates a new and empty compound initializer.
     **/

    public CompoundInitializer() {
        this(s_log);
    }

    /**
     * Creates a new and empty compound initializer that uses the
     * <code>log</code> Logger to log progress through initialization.
     *
     * @param log A logger to be used for logging progress through
     *            initialization.
     **/

    public CompoundInitializer(Logger log) {
        m_log = log;
    }

    /**
     * Adds <code>init</code> to the set of initializers to be
     * initialized by this CompoundInitializer. The most recently
     * added initializers will be invoked last.
     *
     * @param init The initializer to add to this CompoundInitializer
     **/

    public void add(Initializer init) {
        m_inits.add(init);
    }

    /**
     * Implementation of the {@link Initializer#init(DataInitEvent)}
     * method. This implementation proceeds through the list of sub
     * initializers in order and invokes the init(DataInitEvent)
     * method of each sub initializer in turn.
     *
     * @param evt The data init event.
     **/

    public void init(DataInitEvent evt) {
        int i = 1;
        for (Iterator it = m_inits.iterator(); it.hasNext(); i++) {
            Initializer init = (Initializer) it.next();
            if (m_log.isInfoEnabled()) {
                m_log.info("Running Data Init for " + init.getClass() + 
                           " (" + i + " out of " + m_inits.size() + ")");
            }
            init.init(evt);
        }
    }

    /**
     * Implementation of the {@link Initializer#init(DomainInitEvent)}
     * method. This implementation proceeds through the list of sub
     * initializers in order and invokes the init(DomainInitEvent)
     * method of each sub initializer in turn.
     *
     * @param evt The domain init event.
     **/

    public void init(DomainInitEvent evt) {
        int i = 1;
        for (Iterator it = m_inits.iterator(); it.hasNext(); i++) {
            Initializer init = (Initializer) it.next();
            if (m_log.isInfoEnabled()) {
                m_log.info("Running Domain Init for " + init.getClass() + 
                           " (" + i + " out of " + m_inits.size() + ")");
            }
            init.init(evt);
        }
    }

    /**
     * Implementation of the {@link Initializer#init(LegacyInitEvent)}
     * method. This implementation proceeds through the list of sub
     * initializers in order and invokes the init(LegacyInitEvent)
     * method of each sub initializer in turn.
     *
     * @param evt The legacy init event.
     **/

    public void init(LegacyInitEvent evt) {
        int i = 1;
        for (Iterator it = m_inits.iterator(); it.hasNext(); i++) {
            Initializer init = (Initializer) it.next();
            if (m_log.isInfoEnabled()) {
                m_log.info("Running Legacy Init for " + init.getClass() + 
                           " (" + i + " out of " + m_inits.size() + ")");
            }
            init.init(evt);
        }
    }

}
