/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.portal;

import com.arsdigita.kernel.ResourceType;
import com.arsdigita.web.ApplicationType;
import com.arsdigita.domain.DomainObjectInstantiator;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.util.Assert;
import org.apache.log4j.Category;
import java.util.List;
import java.util.Iterator;
import java.util.ArrayList;

/**
 *
 * <p>This class is a convenience class for easily initializing a Portlet.</p>
 * <p>
 * The usage pattern for this class is:
 * <ul>
 * <li> Create a PortletSetup class.</li>
 * <li> Use setters to initialize values.</li>
 * <li> Call the run method ( setup.run() ).</li>
 * </ul>
 * </p>
 * <p>
 * Necessary values that are uninitialized when run() is called throw an 
 * exception. </p>
 * 
 * @author Justin Ross
 * @version $Id: //core-platform/dev/src/com/arsdigita/portal/PortletSetup.java#11 $
 */
public class PortletSetup {
    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/portal/PortletSetup.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    protected String m_profile;
    protected String m_key = null;
    protected String m_typeName = null;
    protected Category m_category;
    protected String m_title = null;
    protected String m_description = null;
    protected ApplicationType m_provider;
    protected String m_stylesheet = null;
    protected DomainObjectInstantiator m_instantiator = null;    


    public PortletSetup(Category category) {
        m_category = category;

    }

    public void setPortletObjectType(String portletObjectType) {
        m_typeName = portletObjectType;
    }

    public void setProfile(String profile) {
        m_profile = profile;
    }

    public void setKey(String key) {
        m_key = key;
    }

    public void setTitle(String title) {
        m_title = title;
    }

    public void setDescription(String description) {
        m_description = description;
    }

    public void setStylesheet(String stylesheet) {
        m_stylesheet = stylesheet;
    }

    public void setInstantiator(DomainObjectInstantiator instantiator) {
        m_instantiator = instantiator;
    }


    public void setProviderApplicationType(ApplicationType provider) {
        m_provider = provider;
    }

    public void setProviderApplicationType(String applicationObjectType) {
        ApplicationType provider =
            ApplicationType.retrieveApplicationTypeForApplication
            (applicationObjectType);

        Assert.assertNotNull(provider, "provider");

        setProviderApplicationType(provider);
    }

    protected void notice(String message) {
        m_category.info("PortletType '" + m_title + "' - " + message);
    }

    public PortletType run() {
        notice("Validating setup...");

        List messages = validate();

        if (messages.size() > 0) {
            Iterator iter = messages.iterator();

            while (iter.hasNext()) {
                m_category.error((String)iter.next());
            }

            return null;
        }

        notice("Done validating.");

        PortletType portletType = process();

        if (portletType != null) {
            portletType.save();
        }
        return portletType;
    }



    protected List validate() {
        ArrayList messages = new ArrayList();

        if (m_title == null)
            messages.add("Title is not set.");
        if (m_typeName == null)
            messages.add("PortletObjectType is not set.");
        if (m_instantiator == null)
            messages.add("Instantiator is not set.");
        if (m_profile == null)
            messages.add("Profile is not set.");

        return messages;
    }

    protected PortletType process() {
        notice("Starting setup...");

        PortletType portletType = null;

        if (!ResourceType.isInstalled(m_typeName)) {
            notice("Not installed.  Installing now...");

            notice("Using the following properties to perform install.");
            notice("  PortletObjectType: " + m_typeName);
            notice("  Title: " + m_title);
            notice("  Description: " + m_description);
            notice("  Profile: " + m_profile);
            notice("  Instantiator: " + m_instantiator);
            notice("  ProviderApplicationType: " + m_provider);
            notice("  Key: " + m_key);
            notice("  StyleSheet: " + m_stylesheet);

            portletType = PortletType.createPortletType
                (m_title, m_profile, m_typeName);

            portletType.setDescription(m_description);

            if (m_provider != null) {
                portletType.setProviderApplicationType(m_provider);
            }
            
            notice("Done installing.");
        } else {
            portletType = PortletType.retrievePortletTypeForPortlet(m_typeName);
        }

        DomainObjectFactory.registerInstantiator(m_typeName, m_instantiator);

        notice("Done setting up.");

        return portletType;
    }
}
