/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.populate.apps;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.log4j.Logger;

import com.arsdigita.persistence.Session;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.TransactionContext;
import com.arsdigita.populate.Utilities;
import com.arsdigita.portal.Portal;
import com.arsdigita.util.Assert;
import com.arsdigita.web.Application;

/**
 * @author bche
 */
public class PopulateApps {
    private static Logger s_log = Logger.getLogger(PopulateApps.class);
    private PopulateAppPairCollection m_PopAppPairs;
    private String m_sBaseStringSeed;
    
    /**
     * Constructor.  returns a new PopulateApps, which populates a collection of applications
     * @param popAppPairs popAppPairs collection for population
     */
    public PopulateApps(PopulateAppPairCollection popAppPairs) {
        m_PopAppPairs = popAppPairs;
    }

    public void setBaseStringSeed(String sSeed) {
        m_sBaseStringSeed = sSeed;
    }
    
    public String getBaseStringSeed() {
        return m_sBaseStringSeed;
    }    
    
    /**
     * Populates a Portal.  Also adds appropriate portlets from the created applications
     * to the portal
     * @param iNumApps number of app instances to create
     * @param appParent the parent application of the apps created
     * @param portal the portal to populate
     * @return a List of BigDecimals which are the ID'sof the apps created during this method      
     */    
    public List populate(int iNumApps, Application appParent, Portal portal) {
        Session ses = SessionManager.getSession();
        TransactionContext txn = ses.getTransactionContext();                                
        
        String sBaseString = Utilities.getBaseString(m_sBaseStringSeed);
        
        int iSize  = m_PopAppPairs.getSize();
        int iTotalSize = iNumApps * iSize;
        s_log.info("Creating " + iNumApps + " instaces of " + iSize + " applications for a total of " +
            iTotalSize + " apps");

        ArrayList apps = new ArrayList(iTotalSize);
                        
        //iterate through populate apps        
        for (int i=0; i < iSize; i++) {
            PopulateAppPair popAppPair = m_PopAppPairs.getPopulateApp(i);            
            PopulateApp popApp = popAppPair.getPopulateApp();
            List args = popAppPair.getArgs();
            
            popApp.setBaseStringSeed(m_sBaseStringSeed);
            
            for (int j=0; j < iNumApps; j++) {
                //remove spaces from the url
                String sTitle = popApp.getAppType().getTitle().replace(' ', '_') + sBaseString + j;
                s_log.info("Creating app " + sTitle + " with args " + args);
                popApp.createApp(sTitle, appParent);
                popApp.populateApp(args);             
                apps.add(popApp.getApp().getID());    
                
                //get the portlet to add if we are populating a portlet
                if (portal != null) {      
                    //put this in a txn so that the updates to portalParent are committed
                    txn.beginTxn();              
                    //refresh view of portal so we can save it        
                    portal = (Portal)Portal.retrieve(portal.getOID());                  
                    //use a cellnumber of 1 for now
                    portal.addPortlet(popApp.getPortlet(), 1);                     
                    portal.save();
                    txn.commitTxn();                                     
                }                       
            }                        
        }
        return apps;            
        
    }
    
    /**
     * creates and populates application instances
     * @param iNumApps the number of application instances to create for each
     * application in this object's popAppPairs collection
     * @param parent the parent app under which to create the new app instances, or null if there is no parent     
     * @return a List of BigDecimals, which are the ID's of the applications created during this method
     */
    public List populate(int iNumApps, Application parent) {        
        return populate(iNumApps, parent, null);
    }
    
    /**
     * Populates a Portal.  Also adds appropriate portlets from the created applications
     * to the portal
     * @param iNumApps number of app instances to create
     * @param parent the portal to populate
     * @return a List of BigDecimals which are the ID'sof the apps created during this method      
     */
    public List populate(int iNumApps, Portal parent) {    
        return populate(iNumApps, null, parent);            
    }
    
    /**
     * Convenience method that creates and populates application instances as root apps
     * @param iNumApps the number of application instances to create for each
     * application in this object's popAppPairs collection
     * @return a List of BigDecimals, which are the ID's of the applications created during this method
     */
    public List populate(int iNumApps) {        
        return populate(iNumApps, null, null);
    }
}
