/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.mail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import javax.activation.DataSource;

/**
 * Provides a simple DataSource that allows in-memory data
 * objects to be attached to a mail message.
 *
 * <p>Based on the sample code distributed with the JavaMail 1.2 API.
 *
 * @author Ron Henderson 
 * @version $Id: //core-platform/dev/src/com/arsdigita/mail/ByteArrayDataSource.java#7 $
 */

public class ByteArrayDataSource implements DataSource {

    /**
     * Holds the data for this DataSource
     */

    private byte[] m_data;

    /**
     * MIME type of the data
     */

    private String m_type;

    /**
     * Name of the data (optional)
     */

    private String m_name;

    /**
     * Creates a data source from an input stream.
     *
     * @param is the InputStream to read from
     * @param type the MIME type of the data
     * @param name the name of the data
     */

    public ByteArrayDataSource (InputStream is,
                                String type,
                                String name)
    {
        m_type = type;
        m_name = name;

        try {
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            int ch;

            while ((ch = is.read()) != -1) {
                os.write(ch);
            }

            m_data = os.toByteArray();
        } catch (IOException ioex) {
            // do something
        }
    }

    /**
     * Creates a data source from a byte array.
     *
     * @param data the byte array containing the data
     * @param type the MIME type of the data
     * @param name the name of the data
     */

    public ByteArrayDataSource (byte[] data,
                                String type,
                                String name)
    {
        m_data = data;
        m_type = type;
        m_name = name;
    }

    /**
     * Creates a data source from a String, assuming the data only
     * contains ASCII characters (iso-8859-1).
     *
     * @param data the String containing the data
     * @param type the MIME type of the data
     * @param name the name of the data
     */

    public ByteArrayDataSource (String data,
                                String type,
                                String name)
    {
        this(data,type,name,"iso-8859-1");
    }

    /**
     * Creates a data source from a String using a specified character
     * set.
     *
     * @param data the String containing the data
     * @param type the MIME type of the data
     * @param name the name of the data
     * @param charset the encoding used for the String
     */

    public ByteArrayDataSource (String data,
                                String type,
                                String name,
                                String charset)
    {
        m_type = type;
        m_name = name;

        try {
            m_data = data.getBytes(charset);
        } catch (UnsupportedEncodingException uex) {
            // do something
        }
    }

    /**
     * Returns an input stream for the data.
     */

    public InputStream getInputStream() throws IOException {
        if (m_data == null) {
            throw new IOException("no data");
        }
        return new ByteArrayInputStream(m_data);
    }

    /**
     * Required by the interface, but not available.
     */

    public OutputStream getOutputStream() throws IOException {
        throw new IOException("not implemented");
    }

    /**
     * Returns the MIME type of the content.
     */

    public String getContentType() {
        return m_type;
    }

    /**
     * Returns the name of the content.
     */

    public String getName() {
        return m_name;
    }
}
