/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel;

import com.arsdigita.kernel.ui.ResourceConfigFormSection;
import com.arsdigita.kernel.ui.BasicResourceConfigFormSection;
import com.arsdigita.kernel.ui.ResourceConfigComponent;
import com.arsdigita.bebop.RequestLocal;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.SaveCancelSection;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSubmissionListener;
import com.arsdigita.bebop.FormProcessException;
import org.apache.log4j.Logger;

/**
 * XXX JAVADOC XXX
 *
 * @see com.arsdigita.kernel.ResourceType
 * @see ResourceConfigFormSection
 * @see com.arsdigita.kernel.Resource
 * @author Justin Ross
 * @version $Id: //core-platform/dev/src/com/arsdigita/kernel/ResourceTypeConfig.java#7 $
 */
public class ResourceTypeConfig {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/kernel/ResourceTypeConfig.java#7 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (ResourceTypeConfig.class);

    /**
     * For use in generating default config components when
     * Resource authors don't specify their own.  Should only be
     * used by PortletType.
     */
    protected ResourceTypeConfig() {
        // Empty
    }

    public ResourceTypeConfig(String resourceObjectType) {
        if (s_log.isDebugEnabled()) {
            s_log.debug("Registering " + this + " to object type " +
                        resourceObjectType);
        }

        ResourceType.registerResourceTypeConfig
            (resourceObjectType, this);
    }

    public ResourceConfigFormSection getCreateFormSection
            (final ResourceType resType, final RequestLocal parentResRL) {
        final BasicResourceConfigFormSection config =
            new BasicResourceConfigFormSection(resType, parentResRL);

        if (s_log.isDebugEnabled()) {
            s_log.debug("Returning default create form section " + config);
        }

        return config;
    }

    public ResourceConfigFormSection getModifyFormSection
            (final RequestLocal application) {
        final BasicResourceConfigFormSection config =
            new BasicResourceConfigFormSection(application);

        return config;
    }

    /**
     * Retrieves the component for creating an instance of a
     * ResourceType. The component should fire a completion
     * event when it has finished processing.
     * @see com.arsdigita.bebop.Completable#fireCompletionEvent(PageState)
     */
    public ResourceConfigComponent getCreateComponent
            (final ResourceType resType, final RequestLocal parentResRL) {
        final ResourceConfigFormSection section =
            getCreateFormSection(resType, parentResRL);

        return new ResourceConfigWrapperComponent(section);
    }

    /**
     * Retrieves the component for modifying an instance of a
     * ResourceType. The component should fire a completion
     * event when it has finished processing.
     * @see com.arsdigita.bebop.Completable#fireCompletionEvent(PageState)
     */
    public ResourceConfigComponent getModifyComponent
            (final RequestLocal resource) {
        final ResourceConfigFormSection section =
            getModifyFormSection(resource);

        return new ResourceConfigWrapperComponent(section);
    }

    public void configureResource(Resource resource) {
        // Empty
    }

    private class ResourceConfigWrapperComponent
            extends ResourceConfigComponent {
        private ResourceConfigFormSection m_section;
        private SaveCancelSection m_buttons;

        public ResourceConfigWrapperComponent
                (ResourceConfigFormSection section) {
            m_section = section;
            m_buttons = new SaveCancelSection();

            Form form = new Form("wrapper");
            form.setRedirecting(true);
            form.add(m_section);
            form.add(m_buttons);

            form.addSubmissionListener(new FormSubmissionListener() {
                    public void submitted(FormSectionEvent e)
                        throws FormProcessException {
                        PageState state = e.getPageState();

                        if (m_buttons.getCancelButton().isSelected(state)) {
                            fireCompletionEvent(state);
                            throw new FormProcessException("cancelled");
                        }
                    }
                });
            form.addProcessListener(new FormProcessListener() {
                    public void process(FormSectionEvent e)
                        throws FormProcessException {
                        PageState state = e.getPageState();
                        fireCompletionEvent(state);
                    }
                });

            add(form);
        }

        public Resource createResource(PageState state) {
            Resource resource = null;
            if (m_buttons.getSaveButton().isSelected(state)) {
                resource = m_section.createResource(state);
            }
            return resource;
        }

        public void modifyResource(PageState state) {
            if (m_buttons.getSaveButton().isSelected(state)) {
                m_section.modifyResource(state);
            }
        }
    }
}
