/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.formbuilder.ui;


import com.arsdigita.bebop.parameters.ParameterData;

import com.arsdigita.bebop.event.ParameterListener;
import com.arsdigita.bebop.event.ParameterEvent;


/**
 * Validates that the parameter submitted is the fully qualified
 * class name of a class that can be loaded. Can also be supplied with a
 * class object and will then additionally check that the submitted class is assignable from the
 * class we are verifying against (see Class.isAssignableFrom(Class), this is useful
 * for checking that the submitted class implements a certain interface or extends a certain
 * class). This validation listener requires that invoking toString() on the value object
 * submitted will yield the fully qualified class name (you may for example use a StringParameter
 * which is the default of most widgets).
 *
 * @author Peter Marklund
 * @version $Id: //core-platform/dev/src/com/arsdigita/formbuilder/ui/ClassValidationListener.java#7 $
 */
public class ClassValidationListener
    implements ParameterListener {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/formbuilder/ui/ClassValidationListener.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private Class m_assignableClass = null;

    /**
     * With this constructor it will only be checked that the submitted class can be
     * loaded.
     */
    public ClassValidationListener() {
        // Intentionally empty
    }

    /**
     * Use this constructor to check that the submitted class can be loaded and that it is
     * assignable from the class that is supplied to this constructor. For example, if you wanted
     * to check that the user-supplied class name belongs to a class that is serializable you
     * would supply the java.io.Serializable class to this constructor.
     */
    public ClassValidationListener(Class assignableClass) {

        m_assignableClass = assignableClass;
    }

    public void validate(ParameterEvent parameterEvent) {
        ParameterData parameterData = parameterEvent.getParameterData();
        Object parameterValue = parameterData.getValue();

        String className = parameterValue.toString();

        Class parameterClass;

        try {
            // Load the class
            parameterClass = Class.forName(className);

        } catch (Exception e) {
            parameterData.addError("The name of the supplied class could not be loaded. Check the name " +
                                   "and make sure you are prefixing the class name with the right package name " +
                                   "(that the class name is fully qualified)");

            // If the class could not be loaded we cannot check assignability
            return;
        }


        if (m_assignableClass != null) {

            if (!m_assignableClass.isAssignableFrom(parameterClass)) {
                parameterData.addError("The supplied class name is not assignable from the class or interface " +
                                       m_assignableClass.getName());
            }
        }
    }
}
