/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.table;

/**
 * The <code>TableModel</code> is the abstraction a {@link
 * com.arsdigita.bebop.Table Table} uses to access the data it
 * displays. The table will ask its {@link TableModelBuilder} to
 * instantiate a new table model once for each request it processes.
 *
 * <p> The table will request each element in the model at most once per
 * request, moving through the rows with successive calls to {@link
 * #nextRow}. For each row, the table retrieves the values and keys in each
 * column with calls to {@link #getElementAt} and {@link #getKeyAt}.
 *
 * <p> The table data is accessed by the table by moving through the rows
 * of the table model with calls to {@link #nextRow}. The data for each
 * column in a row is represented by two objects: the data element which
 * usually contains display information for that column and can be as
 * simple as a string, and the key, which is used to identify the
 * column. The key is usually a suitable representation of the primary key
 * of the underlying object in the database. The key needs to be unique
 * amongst all <em>rows</em> in the table model, but doesn't need to
 * uniquely identify the row <em>and</em> column for that data item -
 * all calls to {@link #getKeyAt} can return the same value for one row in
 * the table model.
 *
 * @see com.arsdigita.bebop.Table Table
 * @see TableModelBuilder
 *
 * @author David Lutterkort
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/table/TableModel.java#7 $ */
public interface TableModel {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/table/TableModel.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * Return the number of columns this table model has.
     *
     * @return the number of columns in the table model
     * @post return >= 0
     */
    int getColumnCount();

    /**
     * Move to the next row and return true if the model is now positioned on
     * a valid row. Initially, the table model is positioned before the first
     * row. The table will call this method before it retrieves the data for
     * the row with calls to {@link #getElementAt getElementAt} and {@link
     * #getKeyAt getKeyAt}.
     *
     * <p> If this method returns <code>true</code>, subsequent calls to
     * {@link #getElementAt getElementAt} and {@link #getKeyAt getKeyAt} have
     * to succeed and return non-null objects. If this method returns
     * <code>false</code>, the table assumes that it has traversed all the
     * data contained in this model.
     *
     * @return <code>true</code> if the model is positioned on a valid row
     */
    boolean nextRow();

    /**
     * Return the data element for the given column and the current row. The
     * returned object will be passed to the table cell renderer as the
     * <code>value</code> argument without modifications.
     *
     * @param columnIndex the number of the column for which to get data
     * @return the object to pass to the table cell renderer for display
     * @pre columnIndex >= 0 && columnIndex < getColumnCount()
     * @post return != null
     * @see TableCellRenderer
     */
    Object getElementAt(int columnIndex);

    /**
     * Return the key for the given column and the current row. The key has
     * to be unique for each <em>row</em> in the table model, but does not
     * need to be unique for each row <em>and</em> column, though it may.
     * The key is passed to the table cell renderer as the <code>key</code>
     * argument.
     *
     * @param columnIndex the number of the column for which to get data
     * @return the key for the given column and the current row.
     * @pre columnIndex >= 0 && columnIndex < getColumnCount()
     * @post return != null
     * @see TableCellRenderer
     */
    Object getKeyAt(int columnIndex);
}
