/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.parameters;

import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.event.ParameterEvent;
import com.arsdigita.bebop.parameters.ParameterData;
import com.arsdigita.globalization.GlobalizedMessage;
import com.arsdigita.util.StringUtils;
import com.arsdigita.util.Assert;

/**
 * <p>
 * Check that a string's length falls into a particular range.
 * </p>
 *
 * @version $Revision: #9 $ $Date: 2004/04/07 $
 */
public class StringInRangeValidationListener extends GlobalizedParameterListener {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/parameters/StringInRangeValidationListener.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private final int m_minLength;
    private final int m_maxLength;

    public StringInRangeValidationListener(int minLength, int maxLength) {

        validateRange(minLength, maxLength);

        m_minLength = minLength;
        m_maxLength = maxLength;
        setErrorText("string_in_range", getBundleBaseName());
    }

    public StringInRangeValidationListener(
                                           int minLength, int maxLength, GlobalizedMessage error
                                           ) {
        validateRange(minLength, maxLength);

        m_minLength = minLength;
        m_maxLength = maxLength;
        setError(error);
    }



    public void validate(ParameterEvent e) throws FormProcessException {
        ParameterData data = e.getParameterData();
        Object obj = data.getValue();

        if (StringUtils.emptyString(obj)) {
            if(m_minLength > 0) {
                data.addError(getError());
            }
            return;
        }

        boolean isValid = true;
        if (obj instanceof String[]) {
            String[] values = (String[]) obj;

            for (int i = 0; i < values.length && isValid; i++) {
                String value = values[i];
                isValid = isInRange(value);
            }
        } else if (obj instanceof String) {
            String value = (String) obj;
            isValid = isInRange(value);
        }

        if (!isValid) {
            data.addError(getError());
        }
    }

    private boolean isInRange(final String value) {
        final int length = value.length();
        final boolean isInRange = length >= m_minLength &&
                length <= m_maxLength;
        return isInRange;
    }

    /**
     *  Sets the error text using a given resource bundle.
     *
     * @param text
     * @param bundle
     */
    private void setErrorText(String text, String bundle) {
        setError(new GlobalizedMessage(
                                       text,
                                       bundle,
                                       new Object[] {new Integer(m_minLength), new Integer(m_maxLength)}
                                       ));
    }


    /**
     * Sanity checks range arguments for constructors.
     *
     * @param minLength
     * @param maxLength
     */
    private static void validateRange(int minLength, int maxLength) {
        Assert.truth(minLength >= 0, "Minimum length cannot be negative!");
        Assert.truth(maxLength > minLength, "Maximum length must be greater than minimum!");
    }


}
