/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.parameters;

import com.arsdigita.globalization.Globalization;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import javax.servlet.http.HttpServletRequest;

/**
 *    A class that represents the model for date and time form parameters.
 *    (based on the code in DateParameter.java)
 *
 *    @author Scot Seago 
 *    @author Uday Mathur 
 *    @version $Id: //core-platform/dev/src/com/arsdigita/bebop/parameters/DateTimeParameter.java#9 $
 */
public class DateTimeParameter extends ParameterModel {

    public DateTimeParameter(String name) {
        super(name);
    }


    /**
     * This method returns a new Calendar object that is manipulated
     * within transformValue to create a Date Object. This method should
     * be overridden if you wish to use a Calendar other than the
     * lenient GregorianCalendar.
     *
     * @param request the servlet request from which Locale can be
     * extracted if needed
     *
     * @return a new Calendar object
     * */
    protected Calendar getCalendar(HttpServletRequest request) {
        return new GregorianCalendar();
    }

    /**
     * Computes a dateTime object from multiple parameters in the
     * request. This method searches for parameters named
     * <code>getName() + ".year"<code>,
     * <code>getName() + ".month"<code>,
     * <code>getName() + ".day"<code>,
     * <code>getName() + ".hour"<code>,
     * <code>getName() + ".minute"<code>,
     * <code>getName() + ".second"<code>, and
     * <code>getName() + ".amOrPm"<code>.
     * */
    public Object transformValue(HttpServletRequest request)
        throws IllegalArgumentException {

        Calendar c = getCalendar(request);
        c.clear();

        String year = Globalization.decodeParameter(request, getName()+".year");
        String month = Globalization.decodeParameter(request, getName()+".month");
        String day = Globalization.decodeParameter(request, getName()+".day");
        String hour = Globalization.decodeParameter(request, getName()+".hour");
        String minute = Globalization.decodeParameter(request, getName()+".minute");
        String second = Globalization.decodeParameter(request, getName()+".second");
        String amOrPm = Globalization.decodeParameter(request, getName()+".amOrPm");

        if ( year == null && month == null && day == null ) {
            return transformSingleValue(request);
        }

        if ( year != null ) {
            c.set(Calendar.YEAR, Integer.parseInt(year));
        }
        if ( month != null ) {
            c.set(Calendar.MONTH, Integer.parseInt(month));
        }
        if ( day != null ) {
            c.set(Calendar.DATE, Integer.parseInt(day));
        }
        if ( hour != null ) {
            c.set(Calendar.HOUR, Integer.parseInt(hour));
        }
        if ( minute != null ) {
            c.set(Calendar.MINUTE, Integer.parseInt(minute));
        }
        if ( second != null ) {
            c.set(Calendar.SECOND, Integer.parseInt(second));
        }
        if ( amOrPm != null ) {
            c.set(Calendar.AM_PM, Integer.parseInt(amOrPm));
        }
        return c.getTime();
    }

    public Object unmarshal(String encoded) {
        try {
            return new Date(Long.parseLong(encoded));
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("Cannot unmarshal dateTime '"
                                               + encoded +"': " + e.getMessage());
        }
    }

    public String marshal(Object value) {
        return Long.toString(((Date) value).getTime());
    }

    public Class getValueClass() {
        return Date.class;
    }

}
