/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.form;

import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.parameters.TimeParameter;
import com.arsdigita.bebop.parameters.NumberInRangeValidationListener;
import com.arsdigita.bebop.parameters.ParameterData;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.bebop.util.BebopConstants;
import com.arsdigita.util.Assert;
import com.arsdigita.xml.Element;

import java.text.DateFormatSymbols;
import java.util.Calendar;


/**
 * A class representing a time field in an HTML form.
 *
 * @see com.arsdigita.bebop.form.DateTime
 * @author Dave Turner
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/form/Time.java#9 $
 */
public class Time extends Widget implements BebopConstants
{

    private TextField m_hour;
    private TextField m_minute;
    private TextField m_second;
    private OptionGroup m_amOrPm;
    private boolean m_showSeconds;
    private static final String ZERO = "0";


    private class HourFragment extends TextField {

        private Time parent;

        public HourFragment(String name, Time parent) {
            super(name);
            this.parent = parent;
            this.addValidationListener(new NumberInRangeValidationListener(1,12));        }

        protected ParameterData getParameterData(PageState ps) {
            Object value = getValue(ps);
            if (value == null) {
                return null;
            }
            return new ParameterData(getParameterModel(), value);
        }

        public Object getValue(PageState ps) {
            return parent.getFragmentValue(ps, Calendar.HOUR);
        }
    }

    private class MinuteFragment extends TextField {

        private Time parent;

        public MinuteFragment(String name, Time parent) {
            super(name);
            this.parent = parent;
            this.addValidationListener(new NumberInRangeValidationListener(0,59));        }

        protected ParameterData getParameterData(PageState ps) {
            Object value = getValue(ps);
            if (value == null) {
                return null;
            }
            return new ParameterData(getParameterModel(), value);
        }

        public Object getValue(PageState ps) {
            Integer min = (Integer) parent.getFragmentValue(ps, Calendar.MINUTE);
            if (min == null) {
                return null;
            }
            if ( min.intValue() < 10 ) {
                return ZERO + min.toString();
            } else {
                return min.toString();
            }
        }
    }

    private class SecondFragment extends TextField {

        private Time parent;

        public SecondFragment(String name, Time parent) {
            super(name);
            this.parent = parent;
            this.addValidationListener(new NumberInRangeValidationListener(0,59));        }

        protected ParameterData getParameterData(PageState ps) {
            Object value = getValue(ps);
            if (value == null) {
                return null;
            }
            return new ParameterData(getParameterModel(), value);
        }

        public Object getValue(PageState ps) {
            Integer sec = (Integer) parent.getFragmentValue(ps, Calendar.SECOND);
            if (sec == null) {
                return null;
            }
            if ( sec.intValue() < 10 ) {
                return ZERO + sec.toString();
            } else {
                return sec.toString();
            }
        }
    }

    private class AmPmFragment extends SingleSelect {

        private Time parent;

        public AmPmFragment(String name, Time parent) {
            super(name);
            this.parent = parent;
        }

        protected ParameterData getParameterData(PageState ps) {
            Object value = getValue(ps);
            if (value == null) {
                return null;
            }
            return new ParameterData(getParameterModel(), value);
        }

        public Object getValue(PageState ps) {
            return parent.getFragmentValue(ps, Calendar.AM_PM);
        }

    }


    /** Constructor. */
    public Time ( ParameterModel model ) {
        this(model, false);
    }

    /** Constructor. */
    public Time ( ParameterModel model, boolean showSeconds ) {
        super(model);

        if ( ! (model instanceof TimeParameter)) {
            throw new IllegalArgumentException (
                                                "The Time widget " + model.getName() +
                                                " must be backed by a TimeParameter parameter model");
        }


        String name = model.getName();
        String nameHour = name + ".hour";
        String nameMinute = name + ".minute";
        String nameSecond = name + ".second";
        String nameAmOrPm = name + ".amOrPm";

        DateFormatSymbols dfs = new DateFormatSymbols();

        m_hour = new HourFragment(nameHour, this);
        m_minute = new MinuteFragment(nameMinute, this);
        m_showSeconds = showSeconds;
        if ( m_showSeconds ) {
            m_second = new SecondFragment(nameSecond, this);
        } else {
            m_second = null;
        }
        m_amOrPm = new AmPmFragment(nameAmOrPm, this);

        m_hour.setMaxLength(2);
        m_hour.setSize(2);
        m_minute.setMaxLength(2);
        m_minute.setSize(2);
        if (m_showSeconds) {
            m_second.setMaxLength(2);
            m_second.setSize(2);
        }

        String [] amPmStrings = dfs.getAmPmStrings();
        for ( int i = 0 ; i < amPmStrings.length ; i++ ) {
            m_amOrPm.addOption(new Option(String.valueOf(i), amPmStrings[i]));
        }

    }


    public Time ( String name ) {
        this(new TimeParameter(name));
    }

    /** Returns a string naming the type of this widget. */
    public String getType () {
        return "time";
    }

    /**
     * Sets the <tt>MAXLENGTH</tt> attributes for the <tt>INPUT</tt> tag
     * used to render this form element.
     */
    public void setMaxLength ( int length ) {
        setAttribute("MAXLENGTH", String.valueOf(length));
    }

    public boolean isCompound () {
        return true;
    }

    /** The XML tag for this derived class of Widget. */
    protected String getElementTag () {
        return "bebop:time";
    }

    public void generateWidget ( PageState ps, Element parent ) {

        if ( ! isVisible(ps) ) {
            return;
        }

        Element time = parent.newChildElement(getElementTag(), BEBOP_XML_NS);
        time.addAttribute("name", getParameterModel().getName());
        m_hour  .generateXML(ps, time);
        m_minute.generateXML(ps, time);
        if ( m_showSeconds ) {
            m_second.generateXML(ps, time);
        }
        m_amOrPm.generateXML(ps, time);
    }

    public void setDisabled () {
        m_hour.setDisabled();
        m_minute.setDisabled();
        if ( m_showSeconds ) {
            m_second.setDisabled();
        }
        m_amOrPm.setDisabled();
    }

    public void setReadOnly () {
        m_hour.setReadOnly();
        m_minute.setReadOnly();
        if ( m_showSeconds ) {
            m_second.setReadOnly();
        }
        m_amOrPm.setReadOnly();
    }


    /**
     * Sets the Form Object for this Widget. This method will throw an
     * exception if the _form pointer is already set. To explicity
     * change the _form pointer the developer must first call
     * setForm(null)
     *
     * @param the <code>Form</code> Object for this Widget.
     * @exception IllegalStateException if form already set.
     */
    public void setForm(Form f) {
        super   .setForm(f);
        m_hour.setForm(f);
        m_minute.setForm(f);
        if (m_showSeconds) {
            m_second.setForm(f);
        }
        m_amOrPm.setForm(f);
    }

    private Object getFragmentValue(PageState ps, int field) {
        Assert.assertNotNull(ps, "PageState");
        FormData f = getForm().getFormData(ps);
        if (f != null) {
            java.util.Date value = (java.util.Date)f.get(getName());
            if (value != null) {
                Calendar c = Calendar.getInstance();
                c.setTime(value);
		int intVal = c.get(field);
		if (field == Calendar.HOUR && intVal == 0) {
		    intVal = 12;
		}
                return new Integer(intVal);
            }
        }
        return null;
    }


}
