/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop;

import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.util.Assert;

/**
 * An implementation of {@link SingleSelectionModel} that uses
 * a state parameter for managing the currently selected key.
 * <p>
 *
 * A typical use case for this class is as follows.
 * <blockquote><pre><code>public TheConstructor() {
 *   m_parameter = new StringParameter("my_key");
 *   m_sel = new ParameterSingleSelectionModel(m_parameter);
 * }
 *
 * public void register(Page p) {
 *   p.addComponent(this);
 *   p.addComponentStateParam(this, m_param);
 * }</code></pre></blockquote>
 *
 * @author Stanislav Freidin
 */
public class ParameterSingleSelectionModel
    extends AbstractSingleSelectionModel {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/ParameterSingleSelectionModel.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private ParameterModel m_parameter;

    /**
     * Constructs a new ParameterSingleSelectionModel.
     *
     * @param m the parameter model that will be used to
     *   keep track of the currently selected key
     */
    public ParameterSingleSelectionModel(ParameterModel m) {
        super();

        m_parameter = m;
    }

    /**
     * Returns the key that identifies the selected element.
     *
     * @param state a <code>PageState</code> value
     * @return a <code>String</code> value.
     */
    public Object getSelectedKey(PageState state) {
        final FormModel model = state.getPage().getStateModel();
        if (model.containsFormParam(m_parameter)) {
            return state.getValue(m_parameter);
        } else {
            return null;
        }
    }

    public final ParameterModel getStateParameter() {
        return m_parameter;
    }

    /**
     * Set the selected key.
     *
     * @param state represents the state of the current request
     * @param newKey the new selected key
     */
    public void setSelectedKey(PageState state, Object newKey) {
        final Object oldKey = getSelectedKey(state);

        if (Assert.isEnabled()) {
            final FormModel model = state.getPage().getStateModel();
            Assert.truth(model.containsFormParam(m_parameter));
        }

        state.setValue(m_parameter, newKey);

        if (newKey == null && oldKey == null) {
            return;
        }

        if (newKey != null && newKey.equals(oldKey)) {
            return;
        }

        fireStateChanged(state);
    }
}
